import React from 'react';
import { Tabs, Table, Spin, Empty } from 'antd';
import { useRequest } from 'umi';
import cls from 'classnames';
import get from 'lodash/get';
import isEmpty from 'lodash/isEmpty';
import * as services from '../../services';
import Modal from '../../components/Modal';
import Button from '../../components/Button';
import Container from '../../components/Container';
import InvokeExe from '../../components/InvokeExe';
import BigData from './BigData';
import SmartMonitor from '../SmartMonitor';
import { AlarmMap, AlarmLevel } from '../../constants';
import Info from './Info';
import styles from './index.less';
import totalLoadRate from '../../../../resource/images/screen/total_load_rate.png';
import realPower from '../../../../resource/images/screen/real_power.png';
import installedCapacity from '../../../../resource/images/screen/installed_capacity.png';
import doneRate from '../../../../resource/images/screen/done_rate.png';
import lockOrange from '@/resource/images/screen/lock_orange.png'
import lockBlue from '@/resource/images/screen/lock_blue.png'
import waterDrop from '@/resource/images/screen/water_drop.png'
import water from '@/resource/images/screen/water.png'
import thermometer from '@/resource/images/screen/thermometer.png'
import temp from '@/resource/images/screen/temp.png'
import smoke from '@/resource/images/screen/smoke.png'
import ChartsMain from '@/pages/screen/Microgrid/components';
import { featureEnum } from '@/pages/screen/constants';

const { TabPane } = Tabs;

const levelsMap = {
  '1': '0.4KV',
  '2': '10KV',
  '6': '20kv',
  '3': '35KV',
  '4': '110KV',
  '5': '220KV',
};

// 资产类型
const assetsTypeMap = {
  1001: '温湿度装置',
  1002: '高压充气开关柜',
  1003: '400V开关柜',
  1101: '触头盒',
  1102: '套管',
  1103: '绝缘支柱',
  1201: '干式变压器',
  1202: '油浸式变压器',
  1301: '有载调压',
  1401: '温控仪',
  1501: '风机',
  1601: '400V电容器柜',
  1701: '控制器',
  1801: '接触器',
  1901: '电抗器',
  2001: '直流屏',
  2002: '通讯屏',
  2003: '故障录波屏',
  2004: '光端机屏',
  2005: '其它辅助屏',
  2006: '辅助屏充电模块',
  2007: '辅助屏电池',
  2008: '辅助屏电源稳压',
  2009: '辅助屏分布式直流电源装置',
  2010: '辅助屏模块',
  2011: '辅助屏母块',
  2012: '辅助屏显示屏',
  2013: '辅助屏蓄电池',
  2101: '三相电能计量柜',
  2201: '馈线名称',
  3001: '多功能表',
  4001: '真空断路器',
  4002: 'SF6断路器',
  4003: '少油断路器',
  4004: '微型断路器',
  4005: '框架断路器',
  4006: '塑壳断路器',
  5001: '避雷器',
  6001: '熔断器',
  7001: '多功能表',
  8001: '隔离开关刀闸',
  8002: '负荷开关刀闸',
  8003: '接地刀闸',
  9001: '开关柜带电显示装置',
  9002: '变压器带电显示',
};

const getIntNum = val => {
  if (!val) {
    return 0;
  }
  return parseInt(val);
};

function Customer(props) {
  const { data: infoData, loading: infoLoading } = useRequest(
    () => services.getCustomerDetail(props._id),
    {
      formatResult(res) {
        return get(res, 'data.bean.customer');
      },
    },
  );

  let info = infoData || {};
  info = Object.assign({}, info, { ...props });
  const tableData = get(info, 'equipments', []);
  const columns = [
    // {
    //   title: 'ID',
    //   dataIndex: 'id',
    //   key: 'id',
    //   width: 50,
    // },
    {
      title: '资产名',
      dataIndex: 'name',
      key: 'name',
      width: 100,
    },
    {
      title: '类型',
      dataIndex: 'type',
      key: 'type',
      width: 130,
      render(key) {
        return assetsTypeMap[key] || key;
      },
    },
    {
      title: '厂商',
      dataIndex: 'factory',
      key: 'factory',
      width: 90,
    },
    {
      title: '资产型号',
      dataIndex: 'model',
      key: 'model',
      width: 100,
    },
    {
      title: '出厂编号',
      dataIndex: 'code',
      key: 'code',
      width: 100,
    },
    {
      title: '出厂日期',
      dataIndex: 'production_date',
      key: 'production_date',
      width: 130,
      // render(key) {
      //   const isError = key === RunStatusEnum.error
      //   return <span className={cls({ [styles.error]: isError })}>{RunStatusMap[key]}</span>
      // }
    },
  ];
  const minHeight = 450;
  const infoEmpty = isEmpty(infoData);
  const error = get(info, 'alarm[0]');

  const onClickVideo = params => {
    props.handleCustomerStation(params);
    props.onClickVideo(true);
  };

  return (
    <Modal className={cls('screen-table', styles.box)}>
      <Tabs>
        <TabPane tab="概况" key="0" style={{ minHeight }}>
          <Container
            loading={infoLoading}
            loadingCls={styles.loading}
            empty={infoEmpty}
            emptyCls={styles.empty}
          >
            {error && (
              <div className={styles.warning}>
                <div className={styles.warningText}>异常</div>
                <div>
                  <div>异常信息：{error.alarm_value}</div>
                  <div>异常开始时间：{error.start_time}</div>
                  <div>异常类型：{AlarmMap[error.alarm_type]}</div>
                  <div>异常种类：{AlarmLevel[error.alarm_level]}</div>
                </div>
              </div>
            )}
            <div className={styles.customerInfo}>
              <div>户号：{get(info, 'number')}</div>
              <div>客户：{get(info, 'customer_name')}</div>
              <div>地址：{get(info, 'addr', '-')}</div>
              <div>
                电压等级：{levelsMap[get(info, 'voltage_level')] || '-'}
              </div>
              <div>
                班组：
                {get(info, 'service_team_name', '-')}
              </div>
            </div>
            <div className={styles.infoBox}>
              <div className={styles.infoRow}>
                <Info
                  img={totalLoadRate}
                  label="实时负荷率%"
                  value={getIntNum(get(info, 'rate') * 100)}
                />
                <Info
                  img={realPower}
                  label="实时功率kW"
                  value={getIntNum(get(info, 'psum'))}
                />
                <Info
                  img={installedCapacity}
                  label="装机容量kVA"
                  value={getIntNum(get(info, 'transformer_capacity'))}
                />
                <Info
                  img={doneRate}
                  label="主变台数"
                  value={get(info, 'transformer_count')}
                />
              </div>
              <div className={styles.infoRow}>
                <Info
                  img={doneRate}
                  label="本月用电量MWh"
                  value={getIntNum(get(info, 'ep'))}
                />
                <Info
                  img={doneRate}
                  label="本月最大需量kW"
                  value={getIntNum(get(info, 'px', 0))}
                />
              </div>
            </div>
            <div className={styles.stationList}>
              {get(info, 'stations', []).map((item, index) => {
                return (
                  <div key={index} className={styles.powerStation}>
                    <div className={styles.stationName}>{item.name}</div>
                    <div className={styles.stationBtns}>
                      <Button
                        className={styles.btn}
                        onClick={() =>
                          onClickVideo(Object.assign({}, item, { info }))
                        }
                      >
                        实时视频
                      </Button>
                      {
                        <Button
                          className={styles.btn}
                          onClick={() =>
                            props.onClickImg(Object.assign({}, item, { info }))
                          }
                        >
                          一次系统图
                        </Button>
                      }
                      {/* <Button
                        className={styles.btn}
                        onClick={() => props.onClickEnergyChart(Object.assign({}, item, {info}))}
                      >
                        智能监控
                      </Button> */}
                      <Button
                        className={styles.btn}
                        onClick={() => props.onClickRunStatus('1')}
                      >
                        运行状态
                      </Button>
                      <Button
                        className={styles.btn}
                        onClick={() => props.onClickCheckInfo('2')}
                      >
                        环境信息
                      </Button>
                      {/* <Button
                        className={styles.btn}
                        onClick={() => props.onClickCheckStatistics(Object.assign({}, item, {info}))}
                      >
                        智能监控
                      </Button> */}
                    </div>
                  </div>
                );
              })}
            </div>
          </Container>
        </TabPane>
        <TabPane tab="智慧能效" key="1" style={{ minHeight }}>
          <SmartMonitor customerInfo={info} />
        </TabPane>
        <TabPane tab="资产清单" key="2" className={styles.table}>
          <Table
            tableLayout="fixed"
            scroll={{ y: 320 }}
            rowKey="id"
            columns={columns}
            dataSource={tableData}
            loading={infoLoading}
            locale={{
              emptyText: '暂无数据',
            }}
          />
        </TabPane>
        <TabPane tab="电气安全" key="3" style={{ minHeight }}>
          {/*<div className="comingsoon">敬请期待</div>*/}
          <div>
            <div className={styles.elecTitle}>环境状态</div>
            <div className={styles.flexRow}>
              <Info
                img={thermometer}
                label="环境温度℃"
                value="72"
                imgStyle={{width: 20, height: 'auto'}}
              />
              <Info
                img={waterDrop}
                label="环境湿度%"
                value="18"
                imgStyle={{width: 20, height: 'auto'}}
              />
              <Info
                img={temp}
                label="低压母排温度℃"
                value="23"
                imgStyle={{width: 20, height: 'auto'}}
              />
            </div>
            <div className={styles.flexRowWithBorder}>
              <Info
                img={lockBlue}
                label="门禁1"
                value="正常"
                imgStyle={{width: 20, height: 'auto'}}
                valueStyle={{fontSize: 25}}
              />
              <Info
                img={lockOrange}
                label="门禁2"
                value="正常"
                imgStyle={{width: 20, height: 'auto'}}
                valueStyle={{fontSize: 25}}
              />
              <Info
                img={smoke}
                label="烟感"
                value="正常"
                imgStyle={{width: 20, height: 'auto'}}
                valueStyle={{fontSize: 25}}
              />
              <Info
                img={water}
                label="水浸"
                value="正常"
                imgStyle={{width: 20, height: 'auto'}}
                valueStyle={{fontSize: 27}}
              />
            </div>
            <div className={styles.elecTitle}>电量监测</div>
            <div className={styles.flexRowWithBorder}>
              <Info
                label="电流超限"
                value="4"
                unit="次"
                unitStyle={{color: '#8c8f90'}}
                valueStyle={{color: '#FC3242'}}
              />
              <Info
                label="电压超限"
                value="4"
                unit="次"
                unitStyle={{color: '#8c8f90'}}
                valueStyle={{color: '#FC3242'}}
              />
              <Info
                label="漏电流"
                value="0"
                unit="次"
                unitStyle={{color: '#8c8f90'}}
              />
            </div>
            {/*<div className={styles.elecTitle}>实时气象</div>*/}
            {/*<div>24～28℃ 晴转多云 东风＜3级 空气质量优</div>*/}
          </div>
        </TabPane>
        <TabPane tab="大数据分析" key="4" style={{ minHeight }}>
          <BigData customerInfo={info} />
        </TabPane>
        <TabPane tab="微电网" key="5" style={{ minHeight }}>
          <div
            className={
              props.isNormal
                ? styles.microgirdChartsNormal
                : styles.microgirdCharts
            }
          >
            <ChartsMain
              {...props}
              isNormal={false}
              isModal
              extra={{
                sumChargingUnit: 'kWh',
                sumDischargeUnit: 'kWh',
                sumProfitUnit: '元',
                theoryProfitUnit: '元',
              }}
            />
            <div className={styles.bottomBtn}>
              {info.features && info.features.includes(featureEnum.eleNet) && (
                <Button
                  className={styles.btn}
                >
                  <a
                    target="_blank"
                    href="https://epkeeper.net/"
                  >
                    电力物业
                  </a>
                </Button>
              )}
              <Button
                  className={styles.btn}
                >
                  <a target="_blank" href="http://www.jz-energy.cn/bz/views/main/default?system_id=3_2&subprojectid=20200011001">储能系统</a>
                </Button>
                <Button
                  className={styles.btn}
                >
                  <InvokeExe />
                </Button>
            </div>
          </div>
        </TabPane>
      </Tabs>
    </Modal>
  );
}

export default React.memo(Customer);
