import json

from rest_framework import serializers

from console.customer.models import Contact
from crm.crm_customer.models import CrmCustomer


class CrmCustomerLSerializer(serializers.Serializer):
    name = serializers.CharField(max_length=128, help_text="名称")
    number = serializers.CharField(max_length=32, help_text="户号")
    type = serializers.ListField(max_length=32, help_text="类型")
    level = serializers.ListField(max_length=32, help_text="等级")
    address = serializers.ListField(max_length=128, help_text="地址")
    longitude = serializers.CharField(max_length=16, help_text="经度")
    latitude = serializers.CharField(max_length=16, help_text="纬度")
    adcode = serializers.CharField(max_length=16, help_text="省级编码")
    city_code = serializers.CharField(max_length=16, help_text="城市编码")
    district = serializers.CharField(max_length=16, help_text="区县编码")
    logo = serializers.ListField(max_length=128, help_text="logo")


class CustomerInfoSerializer(serializers.ModelSerializer):
    ele_info = serializers.SerializerMethodField()
    customer_info = serializers.SerializerMethodField()
    contact = serializers.SerializerMethodField()

    def get_ele_info(self, obj):
        return json.loads(obj.ele_info) if obj.ele_info else None

    def get_contact(self, obj):
        contact = Contact.objects.filter(customer=obj.customer)
        return [{"name": item.name, "phone": item.phone, "tel": item.tel,
                 "email": item.email, "qq": item.qq, "wechat": item.wechat,
                 "is_urge": item.is_urge, "is_quit": item.is_quit} for item in contact]

    def get_customer_info(self, obj):
        return {
            "name": obj.customer.name,
            "level": obj.customer.level,
            "type": obj.customer.type,
            "service_staff": obj.customer.service_staff.nickname,
            "last_service_staff": obj.customer.last_service_staff.nickname,
            "enterprise_name": obj.customer.enterprise.name,
            "file": obj.customer.enterprise.file,
            "address": obj.customer.enterprise.address,
            "adcode": obj.customer.enterprise.adcode,
            "city_code": obj.customer.enterprise.city_code,
            "district": obj.customer.enterprise.district,
            "longitude": obj.customer.enterprise.longitude,
            "latitude": obj.customer.enterprise.latitude,
            "legal_person": obj.customer.enterprise.legal_person,
            "legal_person_phone": obj.customer.enterprise.legal_person_phone,
            "tax_num": obj.customer.enterprise.tax_num,
            "bank_account_name": obj.customer.enterprise.bank_account_name,
            "bank_name": obj.customer.enterprise.bank_name,
            "postcode": obj.customer.enterprise.postcode,
            "scale": obj.customer.enterprise.scale,
            "nature": obj.customer.enterprise.nature,
            "industry": obj.customer.enterprise.industry,
            "asset": obj.customer.enterprise.asset,
            "covered_area": obj.customer.enterprise.covered_area,
            "parent_enterprise": obj.customer.enterprise.parent_enterprise.name
            if obj.customer.enterprise.parent_enterprise else None
        }

    class Meta:
        model = CrmCustomer
        fields = ("id", "ele_info", "customer_info", "contact")


class CrmCustomerClueSerializer(serializers.Serializer):
    name = serializers.CharField(max_length=128, help_text="名称")
    address = serializers.CharField(max_length=128, help_text="地址")
    longitude = serializers.CharField(max_length=16, help_text="经度")
    latitude = serializers.CharField(max_length=16, help_text="纬度")
    adcode = serializers.CharField(max_length=16, help_text="省级编码")
    city_code = serializers.CharField(max_length=16, help_text="城市编码")
    district = serializers.CharField(max_length=16, help_text="区县编码")
    level = serializers.IntegerField(default=0, help_text="客户等级")
    ele_user_count = serializers.IntegerField(default=0, help_text="户号数")
    content = serializers.DictField(help_text="客户线索")


class CrmCustomerClueUSerializer(CrmCustomerClueSerializer):
    id = serializers.IntegerField(help_text="线索id")


class CrmCustomerBaseSerializer(serializers.Serializer):
    id = serializers.IntegerField()
    name = serializers.CharField(max_length=128, help_text="客户名称")
    level = serializers.IntegerField(help_text="客户等级")
    address = serializers.CharField(max_length=128, help_text="地址")
    longitude = serializers.CharField(max_length=16, help_text="经度")
    latitude = serializers.CharField(max_length=16, help_text="纬度")
    adcode = serializers.CharField(max_length=16, help_text="省级编码")
    city_code = serializers.CharField(max_length=16, help_text="城市编码")
    district = serializers.CharField(max_length=16, help_text="区县编码")
    status = serializers.IntegerField(help_text="状态")
    customer_id = serializers.IntegerField(help_text="客户id")


class CrmCustomerClueApproveSerializer(serializers.Serializer):
    salesman_id = serializers.IntegerField(required=False, allow_null=True, help_text="业务员")
    service_enterprise_id = serializers.IntegerField(help_text="服务公司 id")


class ImportUsersSerializer(serializers.Serializer):
    user_list = serializers.ListField(help_text="营销人员列表")
