from django.db import models

from common.utils.models import BaseModel
from common.utils import choices
from common.accounts.models import Enterprise, Users

from console.powerstation.models import PowerStation, Outline
from console.equipment.models import Equipment
from console.customer.models import Customer
from console.electricityuser.models import ElectricityUser
from console.electrical_info.models import ElectricalInfromation


class MonitorTemplates(BaseModel):
    """
        role = [
                {
                    "fields": "current",
                    "range": [100,200],
                    "duration": 60
                },
                {
                    "fields": "voltage",
                    "range": [100,200],
                    "duration": 60
                },
                {
                    "fields": "load",
                    "threshold": 0.8,
                    "duration": 60
                },
            ]
    """

    name = models.CharField(max_length=64, help_text="告警模板")
    role = models.CharField(max_length=1024, help_text="告警规则")
    notification_type = models.CharField(max_length=16, help_text="通知方式")
    comments = models.CharField(max_length=1024, help_text="备注")

    class Meta:
        db_table = "monitor_templates"


class MonitorPoints(BaseModel):
    name = models.CharField(max_length=64, help_text="监控点名称")
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 max_length=32,
                                 null=True,
                                 help_text="客户")
    electricity_user = models.ForeignKey(to=ElectricityUser,
                                         on_delete=models.SET_NULL,
                                         null=True,
                                         help_text="户号")
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="电站")
    electrical_info = models.ForeignKey(to=ElectricalInfromation,
                                        on_delete=models.SET_NULL,
                                        null=True,
                                        help_text="电气信息")
    outline = models.ForeignKey(to=Outline,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="出线侧")
    equipment = models.ForeignKey(to=Equipment,
                                  on_delete=models.SET_NULL,
                                  max_length=32,
                                  null=True,
                                  help_text="设备")
    frequency = models.IntegerField(null=True, help_text="上传频率, 单位秒")
    template = models.ForeignKey(to=MonitorTemplates,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="模板")
    current_ratio = models.FloatField(default=1.00, help_text="流变")
    voltage_ratio = models.FloatField(default=1.00, help_text="压变")
    comments = models.CharField(max_length=1024, null=True, help_text="备注")
    # 0异常、1正常、2断电
    status = models.IntegerField(choices=choices.MONITOR_POINT_STATUS,
                                 default=1,
                                 help_text="电站状态")
    power = models.FloatField(default=0, help_text="额定功率")
    phone_list = models.CharField(max_length=256, null=True)

    class Meta:
        db_table = "monitor_points"


class MonitorDevices(BaseModel):
    manufacturer = models.IntegerField(choices=choices.MONITOR_MANUFACTURER,
                                       null=True,
                                       help_text="制造商")
    model = models.IntegerField(choices=choices.MONITOR_MANUFACTURER,
                                null=True,
                                help_text="监控设备型号")
    imei = models.CharField(max_length=32, null=True, help_text="IMEI号（设备序列号")
    network_type = models.IntegerField(choices=choices.MONITOR_NETWORK_TP,
                                       null=True,
                                       help_text="网络类型")
    sim = models.CharField(max_length=32, null=True, help_text="sim卡号")
    iccid = models.CharField(max_length=32, null=True, help_text="物联网卡号")
    # 1正常、2断电、3 已拆回
    status = models.IntegerField(choices=choices.MONITOR_DEVICE_STATUS,
                                 default=1,
                                 help_text="状态")
    current_ratio = models.FloatField(default=1.00, help_text="流变")
    voltage_ratio = models.FloatField(default=1.00, help_text="压变")
    monitor_point = models.ForeignKey(to=MonitorPoints,
                                      on_delete=models.SET_NULL,
                                      null=True,
                                      related_name="ptdv",
                                      help_text="监控点")
    enterprise = models.ForeignKey(to=Enterprise,
                                   on_delete=models.SET_NULL,
                                   null=True,
                                   related_name="ptdv",
                                   help_text="所属管理者公司")
    from_oa = models.BooleanField(default=False, help_text="继承oa")
    device_img = models.CharField(null=True, blank=True, max_length=256, help_text="照片")
    comments = models.CharField(null=True, blank=True, max_length=1024, help_text="备注")

    class Meta:
        db_table = "monitor_device"


class MonitorDevicesRecord(BaseModel):
    order_code = models.CharField(max_length=32, help_text="工程编号")
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 max_length=32,
                                 null=True,
                                 help_text="客户")
    electricity_user = models.ForeignKey(to=ElectricityUser,
                                         on_delete=models.SET_NULL,
                                         null=True,
                                         help_text="户号")
    electrical_info = models.ForeignKey(to=ElectricalInfromation,
                                        on_delete=models.SET_NULL,
                                        null=True,
                                        help_text="电气信息")
    device = models.ForeignKey(to=MonitorDevices,
                               on_delete=models.SET_NULL,
                               null=True,
                               help_text="监控设备")
    worker = models.ForeignKey(to=Users,
                               on_delete=models.SET_NULL,
                               null=True,
                               related_name="device_record_worker",
                               help_text="工作人id")
    approver = models.ForeignKey(to=Users,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 related_name="device_record_approver",
                                 help_text="审批人id")
    online_time = models.DateTimeField(null=True,
                                       help_text="上线时间")
    approval_time = models.DateTimeField(null=True,
                                         help_text="审批时间")
    meter_img = models.CharField(max_length=256, help_text="电表照片")
    trans_nameplate_img = models.CharField(max_length=256, help_text="变压器铭牌")
    complete_img = models.CharField(max_length=256, help_text="安装完成整体照片")
    data_img = models.CharField(max_length=256, help_text="监控数据截图")
    other_img = models.CharField(max_length=256, null=True, blank=True, help_text="其他照片")
    # 0待上线 1待审批 2 已通过
    status = models.IntegerField(choices=choices.MONITOR_DEVICE_RECORD_STATUS,
                                 default=0,
                                 help_text="状态")
    meter_number = models.CharField(max_length=32, help_text="电表号")
    trans_number = models.CharField(max_length=32, help_text="变压器号")
    trans_capacity = models.IntegerField(help_text="变压器容量")
    name = models.CharField(max_length=64, help_text="点位名称")
    voltage_ratio = models.DecimalField(max_digits=6,
                                        decimal_places=2,
                                        help_text="压变")
    current_ratio = models.DecimalField(max_digits=6,
                                        decimal_places=2,
                                        help_text="流变")
    comments = models.CharField(max_length=1024, null=True, help_text="备注")
    iccid = models.CharField(max_length=32, null=True, help_text="sim卡iccid")

    class Meta:
        db_table = "monitor_device_record"


class MonitorDevicesImportRecord(BaseModel):
    order_id = models.CharField(max_length=36, help_text="订单id")
    task_id = models.CharField(max_length=36, help_text="任务id")
    url = models.CharField(max_length=256, help_text="cos url")
    status = models.SmallIntegerField(default=0, help_text="状态")
    uploader = models.ForeignKey(to=Users,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="上传者id")
    complete_time = models.DateTimeField(null=True)

    class Meta:
        db_table = "monitor_device_import_record"


class MonitorMetaDataForTest(BaseModel):
    imei = models.CharField(max_length=32, null=True, help_text="imei号")
    ua = models.FloatField(null=True, help_text="A相电流")
    ub = models.FloatField(null=True, help_text="B相电流")
    uc = models.FloatField(null=True, help_text="C相电流")

    uab = models.FloatField(null=True, help_text="AB线电压")
    ubc = models.FloatField(null=True, help_text="BC线电压")
    uca = models.FloatField(null=True, help_text="CA线电压")

    ia = models.FloatField(null=True, help_text="A相电流")
    ib = models.FloatField(null=True, help_text="B相电流")
    ic = models.FloatField(null=True, help_text="C相电流")
    ir = models.FloatField(null=True, help_text="剩余电流")

    pa = models.FloatField(null=True, help_text="A相有功功率")
    pb = models.FloatField(null=True, help_text="B相有功功率")
    pc = models.FloatField(null=True, help_text="C相有功功率")
    p = models.FloatField(null=True, help_text="总有功功率")

    qa = models.FloatField(null=True, help_text="A相无功功率")
    qb = models.FloatField(null=True, help_text="B相无功功率")
    qc = models.FloatField(null=True, help_text="C相无功功率")
    q = models.FloatField(null=True, help_text="总无功功率")

    sa = models.FloatField(null=True, help_text="A相视在功率")
    sb = models.FloatField(null=True, help_text="A相视在功率")
    sc = models.FloatField(null=True, help_text="A相视在功率")
    s = models.FloatField(null=True, help_text="A相视在功率")

    pfa = models.FloatField(null=True, help_text="A相功率因数")
    pfb = models.FloatField(null=True, help_text="B相功率因数")
    pfc = models.FloatField(null=True, help_text="C相功率因数")
    pf = models.FloatField(null=True, help_text="总功率因数")

    f = models.FloatField(null=True, help_text="电源频率")

    ept = models.FloatField(null=True, help_text="总有功电能")
    eqi = models.FloatField(null=True, help_text="正向无功电能")
    eqe = models.FloatField(null=True, help_text="反向无功电能")

    p_d = models.FloatField(null=True, help_text="总有功功率实时需量")

    lvur = models.FloatField(null=True, help_text="三相电压不平衡度")
    iur = models.FloatField(null=True, help_text="三相电流不平衡度")

    ua_thd = models.FloatField(null=True, help_text="A相电压总谐波畸变率")
    ub_thd = models.FloatField(null=True, help_text="B相电压总谐波畸变率")
    uc_thd = models.FloatField(null=True, help_text="C相电压总谐波畸变率")

    ia_thd = models.FloatField(null=True, help_text="A相电流总谐波畸变率")
    ib_thd = models.FloatField(null=True, help_text="B相电流总谐波畸变率")
    ic_thd = models.FloatField(null=True, help_text="C相电流总谐波畸变率")

    t1 = models.FloatField(null=True, help_text="温度1")
    t2 = models.FloatField(null=True, help_text="温度2")
    t3 = models.FloatField(null=True, help_text="温度3")
    t4 = models.FloatField(null=True, help_text="温度4")

    tm = models.DateTimeField(null=True, help_text="上传时间")

    segment1 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment2 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment3 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment4 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment5 = models.CharField(max_length=32,
                                default="NA",
                                null=True,
                                help_text="冗余字段")
    segment6 = models.CharField(max_length=32,
                                default="NA",
                                null=True,
                                help_text="冗余字段")

    class Meta:
        db_table = "mo_test"


class MonitorMetaData(BaseModel):
    point_id = models.IntegerField(null=True, help_text="监控点")
    ua = models.FloatField(null=True, help_text="A相电流")
    ub = models.FloatField(null=True, help_text="B相电流")
    uc = models.FloatField(null=True, help_text="C相电流")

    uab = models.FloatField(null=True, help_text="AB线电压")
    ubc = models.FloatField(null=True, help_text="BC线电压")
    uca = models.FloatField(null=True, help_text="CA线电压")

    ia = models.FloatField(null=True, help_text="A相电流")
    ib = models.FloatField(null=True, help_text="B相电流")
    ic = models.FloatField(null=True, help_text="C相电流")
    ir = models.FloatField(null=True, help_text="剩余电流")

    pa = models.FloatField(null=True, help_text="A相有功功率")
    pb = models.FloatField(null=True, help_text="B相有功功率")
    pc = models.FloatField(null=True, help_text="C相有功功率")
    p = models.FloatField(null=True, help_text="总有功功率")

    qa = models.FloatField(null=True, help_text="A相无功功率")
    qb = models.FloatField(null=True, help_text="B相无功功率")
    qc = models.FloatField(null=True, help_text="C相无功功率")
    q = models.FloatField(null=True, help_text="总无功功率")

    sa = models.FloatField(null=True, help_text="A相视在功率")
    sb = models.FloatField(null=True, help_text="A相视在功率")
    sc = models.FloatField(null=True, help_text="A相视在功率")
    s = models.FloatField(null=True, help_text="A相视在功率")

    pfa = models.FloatField(null=True, help_text="A相功率因数")
    pfb = models.FloatField(null=True, help_text="B相功率因数")
    pfc = models.FloatField(null=True, help_text="C相功率因数")
    pf = models.FloatField(null=True, help_text="总功率因数")

    f = models.FloatField(null=True, help_text="电源频率")

    ept = models.FloatField(null=True, help_text="总有功电能")
    eqi = models.FloatField(null=True, help_text="正向无功电能")
    eqe = models.FloatField(null=True, help_text="反向无功电能")

    p_d = models.FloatField(null=True, help_text="总有功功率实时需量")

    lvur = models.FloatField(null=True, help_text="三相电压不平衡度")
    iur = models.FloatField(null=True, help_text="三相电流不平衡度")

    ua_thd = models.FloatField(null=True, help_text="A相电压总谐波畸变率")
    ub_thd = models.FloatField(null=True, help_text="B相电压总谐波畸变率")
    uc_thd = models.FloatField(null=True, help_text="C相电压总谐波畸变率")

    ia_thd = models.FloatField(null=True, help_text="A相电流总谐波畸变率")
    ib_thd = models.FloatField(null=True, help_text="B相电流总谐波畸变率")
    ic_thd = models.FloatField(null=True, help_text="C相电流总谐波畸变率")

    t1 = models.FloatField(null=True, help_text="温度1")
    t2 = models.FloatField(null=True, help_text="温度2")
    t3 = models.FloatField(null=True, help_text="温度3")
    t4 = models.FloatField(null=True, help_text="温度4")
    tm = models.DateTimeField(null=True, help_text="上传时间")

    segment1 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment2 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment3 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment4 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment5 = models.CharField(max_length=32,
                                default="NA",
                                null=True,
                                help_text="冗余字段")
    segment6 = models.CharField(max_length=32,
                                default="NA",
                                null=True,
                                help_text="冗余字段")

    class Meta:
        abstract = True


class MonitorAlarm(BaseModel):
    name = models.CharField(max_length=64, help_text="告警内容")
    type = models.SmallIntegerField(choices=choices.MONITOR_ALARM_TYPE,
                                    help_text="告警类型")
    level = models.SmallIntegerField(choices=choices.MONITOR_ALARM_LEVEL,
                                     help_text="告警等级")
    code = models.CharField(max_length=16, help_text="告警码")
    duration = models.IntegerField(help_text="持续时间(分钟)")
    status = models.SmallIntegerField(choices=choices.MONITOR_ALARM_STATUS,
                                      help_text="处理状态")
    monitor_point = models.ForeignKey(to=MonitorPoints,
                                      on_delete=models.SET_NULL,
                                      null=True,
                                      help_text="监控点")
    is_end = models.BooleanField(default=False, help_text="告警结束")
    is_send = models.BooleanField(default=False, help_text="是否发送告警短信")
    is_alarm = models.BooleanField(default=False, help_text="是否发生告警")
    comments = models.CharField(max_length=1024, help_text="备注")
    tm = models.DateTimeField()

    class Meta:
        db_table = "monitor_alarm"


class MonitorRealData(BaseModel):
    imei = models.CharField(max_length=32, null=True, help_text="imei号")
    ua = models.FloatField(null=True, help_text="A相电流")
    ub = models.FloatField(null=True, help_text="B相电流")
    uc = models.FloatField(null=True, help_text="C相电流")

    uab = models.FloatField(null=True, help_text="AB线电压")
    ubc = models.FloatField(null=True, help_text="BC线电压")
    uca = models.FloatField(null=True, help_text="CA线电压")

    ia = models.FloatField(null=True, help_text="A相电流")
    ib = models.FloatField(null=True, help_text="B相电流")
    ic = models.FloatField(null=True, help_text="C相电流")
    ir = models.FloatField(null=True, help_text="剩余电流")

    pa = models.FloatField(null=True, help_text="A相有功功率")
    pb = models.FloatField(null=True, help_text="B相有功功率")
    pc = models.FloatField(null=True, help_text="C相有功功率")
    p = models.FloatField(null=True, help_text="总有功功率")

    qa = models.FloatField(null=True, help_text="A相无功功率")
    qb = models.FloatField(null=True, help_text="B相无功功率")
    qc = models.FloatField(null=True, help_text="C相无功功率")
    q = models.FloatField(null=True, help_text="总无功功率")

    sa = models.FloatField(null=True, help_text="A相视在功率")
    sb = models.FloatField(null=True, help_text="A相视在功率")
    sc = models.FloatField(null=True, help_text="A相视在功率")
    s = models.FloatField(null=True, help_text="A相视在功率")

    pfa = models.FloatField(null=True, help_text="A相功率因数")
    pfb = models.FloatField(null=True, help_text="B相功率因数")
    pfc = models.FloatField(null=True, help_text="C相功率因数")
    pf = models.FloatField(null=True, help_text="总功率因数")

    f = models.FloatField(null=True, help_text="电源频率")

    ept = models.FloatField(null=True, help_text="总有功电能")
    eqi = models.FloatField(null=True, help_text="正向无功电能")
    eqe = models.FloatField(null=True, help_text="反向无功电能")

    p_d = models.FloatField(null=True, help_text="总有功功率实时需量")

    lvur = models.FloatField(null=True, help_text="三相电压不平衡度")
    iur = models.FloatField(null=True, help_text="三相电流不平衡度")

    ua_thd = models.FloatField(null=True, help_text="A相电压总谐波畸变率")
    ub_thd = models.FloatField(null=True, help_text="B相电压总谐波畸变率")
    uc_thd = models.FloatField(null=True, help_text="C相电压总谐波畸变率")

    ia_thd = models.FloatField(null=True, help_text="A相电流总谐波畸变率")
    ib_thd = models.FloatField(null=True, help_text="B相电流总谐波畸变率")
    ic_thd = models.FloatField(null=True, help_text="C相电流总谐波畸变率")

    t1 = models.FloatField(null=True, help_text="温度1")
    t2 = models.FloatField(null=True, help_text="温度2")
    t3 = models.FloatField(null=True, help_text="温度3")
    t4 = models.FloatField(null=True, help_text="温度4")
    tm = models.DateTimeField(null=True, help_text="上传时间")

    segment1 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment2 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment3 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment4 = models.FloatField(default=0, null=True, help_text="冗余字段")
    segment5 = models.CharField(max_length=32, default="NA", null=True, help_text="冗余字段")
    segment6 = models.CharField(max_length=32, default="NA", null=True, help_text="冗余字段")

    class Meta:
        db_table = "mo_real_data"
