from django.db import models

# Create your models here.
from common.accounts.models import Users, Accounts
from common.utils.models import BaseModel, MyFloatField

from console.customer.models import Customer
from console.powerstation.models import PowerStation
from console.team.models import Team
from console.powerstation.models import Outline


class TeamMember(BaseModel):
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="班组")
    member = models.ForeignKey(to=Users,
                               on_delete=models.SET_NULL,
                               null=True,
                               help_text="班组成员",
                               related_name="team_member")

    class Meta:
        db_table = 'team_number'


class InspectionPlan(BaseModel):
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="电站")
    plan_date = models.DateTimeField(help_text="计划日期")
    leader = models.ForeignKey(to=Users,
                               on_delete=models.SET_NULL,
                               null=True,
                               help_text="客户代表",
                               related_name="spect_leader")
    status = models.CharField(max_length=32, help_text="状态")

    class Meta:
        db_table = 'inspection_plan'


class InspectionTask(BaseModel):
    name = models.CharField(max_length=64, help_text="名称")
    plan = models.ForeignKey(to="InspectionPlan",
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="所属巡检任务")
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="班组")

    status = models.CharField(max_length=32, null=True, help_text="状态")
    tag = models.CharField(max_length=32, null=True, help_text="标签")
    work_date = models.DateTimeField(help_text="巡检日期", null=True)
    assign_date = models.DateTimeField(null=True, help_text="分派日期")
    start_time = models.DateTimeField(null=True, help_text="开始时间")
    end_time = models.DateTimeField(null=True, help_text="结束时间")
    confirm = models.BooleanField(default=False, help_text="客户确认")
    TYPE = ((0, "正常"), (1, "值班站"))
    type = models.CharField(max_length=32,
                            default=0,
                            help_text="巡检任务类型",
                            choices=TYPE)
    inspector = models.CharField(max_length=32, null=True, help_text="巡检员")
    confirm = models.BooleanField(default=False, help_text="客户确认")

    class Meta:
        db_table = 'inspection_task'


class InspectionTaskRecord(BaseModel):
    inspection_task = models.ForeignKey(to="InspectionTask",
                                        on_delete=models.SET_NULL,
                                        null=True,
                                        help_text="所属巡检任务")
    content = models.TextField(help_text="操作内容")
    operator = models.ForeignKey(to=Accounts,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="操作人员")

    class Meta:
        db_table = 'inspection_task_record'


class StationBuilding(BaseModel):
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="所属电站")
    cable_conduit_status = models.CharField(max_length=32,
                                            null=True,
                                            help_text="电缆沟及盖板状态")
    frame_status = models.CharField(max_length=32, null=True, help_text="护网状态")
    ground_status = models.CharField(max_length=32,
                                     null=True,
                                     help_text="地面状态")
    window_status = models.CharField(max_length=32,
                                     null=True,
                                     help_text="门窗状态")
    house_status = models.CharField(max_length=32,
                                    null=True,
                                    help_text="房屋渗水情况")
    cable_holes_status = models.CharField(max_length=32,
                                          null=True,
                                          help_text="电缆夹层孔洞情况")

    class Meta:
        db_table = 'station_building'


class SafetyEquipment(BaseModel):
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="所属电站")
    electroprobe_status = models.CharField(max_length=32,
                                           null=True,
                                           help_text="高压验电笔（1年）")
    es_check_date = models.DateField(null=True, help_text="高压验电笔上次检测日期")
    ground_wire = models.CharField(max_length=32,
                                   null=True,
                                   help_text="接地线（4年）")
    gw_check_date = models.DateField(null=True, help_text="接地线上次检查日期")
    insulating_mat = models.CharField(max_length=32,
                                      null=True,
                                      help_text="绝缘毯（1年）")
    im_check_date = models.DateField(null=True, help_text="绝缘毯上次检查日期")
    insulating_gloves = models.CharField(max_length=32,
                                         null=True,
                                         help_text="绝缘手套（半年）")
    ig_check_date = models.DateField(null=True, help_text="绝缘手套上次检查日期")
    insulating_shoes = models.CharField(max_length=32,
                                        null=True,
                                        help_text="绝缘鞋（半年）")
    is_check_date = models.DateField(null=True, help_text="绝缘鞋上次检查日期")
    extinguisher = models.CharField(max_length=32,
                                    null=True,
                                    help_text="灭火器压力（2年）")
    ex_check_date = models.DateField(null=True, help_text="灭火器上次检查日期")

    class Meta:
        db_table = 'safety_equipment'


class InspectionData(BaseModel):
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="所属电站",
                                related_name="data_station")
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="班组")
    inspection_task = models.ForeignKey(to=InspectionTask,
                                        on_delete=models.SET_NULL,
                                        null=True,
                                        help_text="所属任务")
    temperature = models.FloatField(null=True, help_text="温度")
    humidity = models.FloatField(null=True, help_text="湿度")
    weather = models.CharField(max_length=32, null=True, help_text="天气")
    remarks = models.TextField(null=True, help_text="备注")
    file = models.TextField(null=True, help_text="热成像")
    image = models.TextField(null=True, help_text="照片")
    battery_voltage = models.CharField(max_length=32, null=True, help_text="电池电压")
    direct_voltage = models.CharField(max_length=32, null=True, help_text="母控电压")
    thermal_number = models.CharField(max_length=1024,
                                      null=True,
                                      help_text="热成像温度")
    inspector = models.CharField(max_length=64, null=True, help_text="巡检员")

    class Meta:
        db_table = 'inspection_data'


class PowerInspectionData(BaseModel):
    inspection_task = models.ForeignKey(to=InspectionTask,
                                        on_delete=models.SET_NULL,
                                        null=True,
                                        help_text="所属巡检任务")
    power_number = models.CharField(max_length=32, help_text="电源编号", null=True)
    meter_number = models.CharField(max_length=32, help_text="电表编号", null=True)
    multiplying_power = MyFloatField(null=True, help_text="倍率")
    power_factor = MyFloatField(null=True, help_text="功率因素")
    total_active_power = MyFloatField(null=True, help_text="总有功")
    peak = MyFloatField(null=True, help_text="峰")
    flat_1 = MyFloatField(null=True, help_text="平1")
    flat_2 = MyFloatField(null=True, help_text="平2")
    valley = MyFloatField(null=True, help_text="谷")
    peak_md = MyFloatField(null=True, help_text="峰md")
    flat_1_md = MyFloatField(null=True, help_text="平1md")
    flat_2_md = MyFloatField(null=True, help_text="平2MD")
    valley_md = MyFloatField(null=True, help_text="谷MD")
    max_md = MyFloatField(null=True, help_text="最大MD")
    declare_md = MyFloatField(null=True, help_text="申报MD")
    total_capacity = MyFloatField(null=True, help_text="总容量")
    voltage_level = models.CharField(max_length=32,
                                     null=True,
                                     help_text="电压等级")
    real_capacity = MyFloatField(null=True, help_text="实际容量")
    real_power_factor = MyFloatField(null=True, help_text="实际功率因素")
    reactive_power_1 = MyFloatField(null=True, help_text="无功1")
    reactive_power_2 = MyFloatField(null=True, help_text="无功2")
    pressure_check = models.CharField(max_length=32,
                                      null=True,
                                      help_text="失压检测")

    class Meta:
        db_table = 'power_inspection_data'


class InspectIn(BaseModel):
    inspect_task = models.ForeignKey(to=InspectionTask,
                                     on_delete=models.SET_NULL,
                                     null=True,
                                     help_text="所属任务")
    power_number = models.CharField(max_length=32, help_text="电源编号", null=True)
    v_ab = models.CharField(max_length=16, null=True, help_text="进线侧A相电压")
    v_bc = models.CharField(max_length=16, null=True, help_text="b相电压")
    v_ca = models.CharField(max_length=16, null=True, help_text="c相电压")
    i_a = models.CharField(max_length=16, null=True, help_text="a相电流")
    i_b = models.CharField(max_length=16, null=True, help_text="b相电流")
    i_c = models.CharField(max_length=16, null=True, help_text="c相电流")
    monitor_a = models.CharField(max_length=16, null=True, help_text="a显示器灯")
    monitor_b = models.CharField(max_length=16, null=True, help_text="b显示器灯")
    monitor_c = models.CharField(max_length=16, null=True, help_text="c显示器灯")

    class Meta:
        db_table = 'inspection_in'


class InspectOut(BaseModel):
    inspect_task = models.ForeignKey(to=InspectionTask,
                                     on_delete=models.SET_NULL,
                                     null=True,
                                     help_text="所属任务")
    power_number = models.CharField(max_length=32, help_text="电源编号", null=True)
    power_station = models.ForeignKey(to=PowerStation,
                                      on_delete=models.SET_NULL,
                                      null=True,
                                      db_column="power_station")
    outline = models.ForeignKey(to=Outline,
                                help_text="出线侧设备",
                                null=True,
                                on_delete=models.SET_NULL,
                                db_column="outline")
    o_ia = models.CharField(max_length=16, null=True, help_text="出线侧电流表a")
    o_ib = models.CharField(max_length=16, null=True, help_text="出线侧电流表b")
    o_ic = models.CharField(max_length=16, null=True, help_text="出线侧电流表c")
    cos = MyFloatField(null=True, help_text="相位角，考核因素")
    power = MyFloatField(null=True, help_text="有功")
    monitor_a = models.CharField(max_length=32, null=True, help_text="a显示器灯")
    monitor_b = models.CharField(max_length=32, null=True, help_text="b显示器灯")
    monitor_c = models.CharField(max_length=32, null=True, help_text="c显示器灯")
    voice = models.CharField(max_length=32, null=True, help_text="运行声音")
    fan = models.CharField(max_length=32, null=True, help_text="风扇运行")
    temperature = models.CharField(max_length=64, null=True, help_text="温度")
    oil_leak = models.CharField(max_length=32, null=True, help_text="油位及漏油情况")
    dry = models.CharField(max_length=32, null=True, help_text="干燥剂")
    abnormal = models.CharField(max_length=32, null=True, help_text="异常")
    switch_v_ab = models.CharField(max_length=16,
                                   null=True,
                                   help_text="总开关侧a相电压")
    switch_v_bc = models.CharField(max_length=16,
                                   null=True,
                                   help_text="总开关侧b相电压")
    switch_ia = models.CharField(max_length=16,
                                 null=True,
                                 help_text="总开关侧电流表a")
    switch_ib = models.CharField(max_length=16,
                                 null=True,
                                 help_text="总开关侧电流表b")
    switch_v_ca = models.CharField(max_length=16,
                                   null=True,
                                   help_text="总开关侧c相电压")
    switch_ic = models.CharField(max_length=16,
                                 null=True,
                                 help_text="总开关侧电流表c")
    GGJ = models.CharField(max_length=32,
                           help_text="电容柜",
                           null=True,
                           blank=True)
    temperature_a = MyFloatField(null=True, help_text="变压器三相温度")
    temperature_b = MyFloatField(null=True, help_text="变压器三相温度")
    temperature_c = MyFloatField(null=True, help_text="变压器三相温度")

    class Meta:
        db_table = 'inspection_out'


class Defect(BaseModel):
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="所属客户")
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="所属电站")
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="班组")
    spect_task = models.ForeignKey(to=InspectionTask,
                                   on_delete=models.SET_NULL,
                                   null=True,
                                   help_text="所属巡检任务")
    status = models.CharField(max_length=32, null=True, help_text="状态")
    name = models.CharField(max_length=32, null=True, help_text="名称")
    content = models.CharField(max_length=256, null=True, help_text="内容")
    proposal = models.TextField(null=True, help_text="建议措施")
    level = models.CharField(max_length=16, null=True, help_text="缺陷等级")
    possible_result = models.CharField(max_length=256,
                                       null=True,
                                       help_text="可能后果")
    file = models.TextField(null=True, help_text="图片url")
    follow_up = models.TextField(null=True, help_text="跟进情况")
    remark = models.TextField(null=True, help_text="备注")
    type = models.CharField(max_length=32,
                            null=True,
                            blank=True,
                            help_text="缺陷类型")

    class Meta:
        db_table = 'defect'


class InspectionReport(BaseModel):
    task = models.ForeignKey(to=InspectionTask, on_delete=models.SET_NULL, null=True)
    url = models.CharField(max_length=256, null=True)

    class Meta:
        db_table = 'inspection_report'
