from datetime import datetime

from django.db.models import Q
from django_filters import rest_framework as filter

from console.inspection.models import InspectionPlan
from console.powerstation.models import PowerStation


class InspectionStationFilter(filter.FilterSet):
    leader = filter.NumberFilter(method="get_leader")
    month = filter.CharFilter(method="get_month")

    def get_leader(self, queryset, name, value):
        if value:
            queryset = queryset.filter(leader_id=value)
        return queryset

    def get_month(self, queryset, name, value):
        if value:
            date = datetime.strptime(value, "%Y-%m")
            year = date.year
            month = date.month
            queryset = queryset.filter(plan_date__year=year,
                                       plan_date__month=month)
        return queryset


class TaskFilter(filter.FilterSet):
    status = filter.CharFilter(method="get_status")
    team = filter.NumberFilter(method="get_team")
    name = filter.CharFilter(method="get_name")
    customer_id = filter.NumberFilter(method="get_customer")
    keyword = filter.CharFilter(method="get_keyword")
    station_id = filter.NumberFilter(method="get_station")
    year_month = filter.CharFilter(method="get_year_month")

    def get_year_month(self, queryset, name, value):
        if value:
            year = value.split("-")[0]
            month = value.split("-")[1]
            queryset = queryset.filter(work_date__year=year,
                                       work_date__month=month)
        return queryset

    def get_station(self, queryset, name, value):
        if queryset:
            queryset = queryset.filter(plan__station_id=value)
        return queryset

    def get_keyword(self, queryset, name, v):
        if v:
            queryset = queryset.filter(
                Q(name__icontains=v) | Q(plan__station__name__icontains=v)
                | Q(plan__station__customer__name__icontains=v) |
                Q(plan__station__electricity_user__customer__name__icontains=v)
                | Q(team__team_headman__nickname__icontains=v)
                | Q(plan__station__addr__icontains=v))
        return queryset

    def get_status(self, queryset, name, value):
        if value:
            queryset = queryset.filter(status=value)
            if value == "completed":
                queryset = queryset.order_by("-end_time")
        return queryset

    def get_team(self, queryset, name, value):
        if value:
            if value == "null":
                queryset = queryset.filter(team_id=None)
            else:
                queryset = queryset.filter(team_id=value)
        return queryset

    def get_name(self, queryset, name, value):
        if value:
            queryset = queryset.filter(
                Q(name__contains=value)
                | Q(plan__station__customer__name__contains=value))
        return queryset

    def get_customer(self, queryset, name, value):
        if value:
            stations = PowerStation.objects.filter(
                Q(customer_id=value) | Q(electricity_user__customer_id=value))
            spect_plans = InspectionPlan.objects.filter(station__in=stations)
            queryset = queryset.filter(plan__in=spect_plans)
        return queryset


class DefectFilter(filter.FilterSet):
    status = filter.CharFilter(method="get_status")
    name = filter.CharFilter(method="get_name")

    def get_status(self, queryset, name, value):
        if value:
            status = {"true": "completed", "false": "pending"}
            queryset = queryset.filter(status=status.get(value))
        return queryset

    def get_name(self, queryset, name, value):
        if value:
            queryset = queryset.filter(Q(name__contains=value) |
                                       Q(content__icontains=value) |
                                       Q(customer__name__icontains=value) |
                                       Q(team__name__icontains=value))
        return queryset


class InspectionPlanFilter(filter.FilterSet):
    leader = filter.NumberFilter(method="get_leader")
    date = filter.CharFilter(method="get_month")

    def get_leader(self, queryset, name, value):
        if value:
            queryset = queryset.filter(leader_id=value)
        return queryset

    def get_month(self, queryset, name, value):
        if value:
            queryset = queryset.filter(plan_date__date=value)
        return queryset


class TaskRecentFilter(filter.FilterSet):
    status = filter.CharFilter(method="get_status")
    team = filter.NumberFilter(method="get_team")
    name = filter.CharFilter(method="get_name")
    customer_id = filter.NumberFilter(method="get_customer")
    keyword = filter.CharFilter(method="get_keyword")
    station_id = filter.NumberFilter(method="get_station")
    year_month = filter.CharFilter(method="get_year_month")

    def get_year_month(self, queryset, name, value):
        if value:
            year = value.split("-")[0]
            month = value.split("-")[1]
            queryset = queryset.filter(work_date__year=int(year),
                                       work_date__month=int(month))
        return queryset

    def get_station(self, queryset, name, value):
        if queryset:
            queryset = queryset.filter(plan__station_id=value)
        return queryset

    def get_keyword(self, queryset, name, v):
        if v:
            queryset = queryset.filter(
                Q(name__icontains=v) | Q(plan__station__name__icontains=v)
                | Q(plan__station__customer__name__icontains=v) |
                Q(plan__station__electricity_user__customer__name__icontains=v)
                | Q(team__team_headman__nickname__icontains=v)
                | Q(plan__station__addr__icontains=v))
        return queryset

    def get_status(self, queryset, name, value):
        if value:
            queryset = queryset.filter(status=value)
            if value == "completed":
                # now = datetime.now()
                # year = now.year
                # month = now.month
                # day = now.day
                # if day > 25:
                #     month += 1
                #     if month > 12:
                #         year += 1
                #         month = month % 12
                # queryset = queryset.filter(work_date__year=year,
                #                            work_date__month=month)
                queryset = queryset.order_by("-end_time")
        return queryset

    def get_team(self, queryset, name, value):
        if value:
            queryset = queryset.filter(team_id=value)
            return queryset
        return []

    def get_name(self, queryset, name, value):
        if value:
            queryset = queryset.filter(
                Q(name__contains=value)
                | Q(plan__station__customer__name__contains=value))
        return queryset

    def get_customer(self, queryset, name, value):
        if value:
            stations = PowerStation.objects.filter(
                Q(customer_id=value) | Q(electricity_user__customer_id=value))
            spect_plans = InspectionPlan.objects.filter(station__in=stations)
            queryset = queryset.filter(plan__in=spect_plans)
        return queryset


class DefectExportFilter(filter.FilterSet):
    team_id = filter.CharFilter(method="get_team_id")

    def get_team_id(self, queryset, name, value):
        if value:
            queryset = queryset.filter(team_id=value)
        return queryset
