from django.db import models

# Create your models here.
from common.accounts.models import Enterprise, Users, Tags
from console.organizations.models import Organization
from common.utils.models import BaseModel


class Customer(BaseModel):
    """
    客户表
    """

    enterprise = models.ForeignKey(to=Enterprise,
                                   on_delete=models.SET_NULL,
                                   related_name="customer_enterprise",
                                   null=True,
                                   help_text="公司")
    name = models.CharField(max_length=64, help_text="名称")
    level = models.CharField(max_length=32, null=True, help_text="客户等级")
    # 0 普通 ，1 托管，10 监控，2 意向客户（新进系统的未签约客户） 列表形式存储
    type = models.CharField(max_length=16, null=True, default=0, help_text="客户类型")
    code = models.CharField(max_length=32, null=True, help_text="客户编码")
    service_enterprise = models.ForeignKey(
        to=Enterprise,
        on_delete=models.SET_NULL,
        related_name="customer_service_enterprise",
        null=True,
        help_text="所属管理公司")
    service_organization = models.ForeignKey(to=Organization,
                                             on_delete=models.SET_NULL,
                                             null=True,
                                             help_text="所属管理部门")
    service_staff = models.ForeignKey(to=Users,
                                      on_delete=models.SET_NULL,
                                      related_name='service_staff',
                                      null=True,
                                      help_text="客户代表")
    last_service_staff = models.ForeignKey(to=Users,
                                           on_delete=models.SET_NULL,
                                           related_name='last_service_staff',
                                           null=True,
                                           help_text="原始客户代表")

    class Meta:
        db_table = 'customer'

    def __str__(self):
        return self.name


class CustomerAdmin(BaseModel):
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 related_name='customer_admin',
                                 null=True)
    user = models.ForeignKey(to=Users, on_delete=models.SET_NULL, null=True)

    class Meta:
        db_table = 'customer_admin'

    def __str__(self):
        return self.user.nickname


class Contact(BaseModel):
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 related_name='customer_contacts',
                                 null=True)
    name = models.CharField(max_length=64, help_text="名称")
    phone = models.CharField(max_length=16, help_text="手机号")
    tel = models.CharField(max_length=64, help_text="座机")
    email = models.EmailField(help_text="邮箱")
    qq = models.CharField(max_length=32, help_text="qq号")
    wechat = models.CharField(max_length=64, help_text="微信号")
    is_urge = models.BooleanField(default=False, help_text="是否为催款联系人 0/否 1/是")
    is_quit = models.BooleanField(default=False, help_text="是否离职 0/否 1/是")
    comments = models.CharField(max_length=1024, null=True, blank=True, help_text="备注")
    tags = models.ManyToManyField(
        Tags,
        through='ContactTags',
        through_fields=('contact', 'tag'),
    )

    class Meta:
        db_table = 'contact'


class ContactTags(BaseModel):
    contact = models.ForeignKey(to=Contact,
                                help_text=u'用户id',
                                on_delete=models.SET_NULL,
                                null=True,
                                related_name="contact_tags")
    tag = models.ForeignKey(to=Tags,
                            help_text=u'标签id',
                            on_delete=models.SET_NULL,
                            null=True,
                            related_name="tag_contacts")

    class Meta:
        db_table = "contact_tags"


class CustomerReport(BaseModel):
    customer = models.ForeignKey(to=Customer, on_delete=models.SET_NULL, null=True)
    url = models.CharField(max_length=256, null=True)
    year_month = models.CharField(max_length=16, null=True)
    report_id = models.CharField(max_length=16, null=True)
    power_number = models.CharField(max_length=32, null=True, help_text="户号")
    basic_confirm = models.BooleanField(default=False, help_text="基础报告确认状态")
    basic_confirm_time = models.DateTimeField(null=True, help_text="基础报告确认时间")
    supper_confirm = models.BooleanField(default=False, help_text="升级版报告确认状态")
    supper_confirm_time = models.DateTimeField(null=True, help_text="升级版报告确认时间")

    class Meta:
        db_table = 'customer_report'
