from ep_fix import fix_ep


def get_hour_ele(ep_raw):
    """
    通过输入电能的时间序列ep_raw，对电能数据进行修正后，计算每小时的用电量hour_ele并返回（仅计算电能值跨度为完整小时的用电量）。

    如果ep_raw存在以下两种情况，则返回值为None：
    1. ep_raw的有效数据点（即非空值点）小于2个，无法计算用电量。
    2. 负向电能值超过1000个。（负向电能值指后一个电能值比前一个电能值小）

    :param ep_raw: pd.Series
            原始电能数据时间序列。index为时间，data为电能值：
                index
                2020-09-04 23:35:24    8603.40
                2020-09-04 23:40:22    8612.40
                2020-09-04 23:45:20    8620.20
                2020-09-04 23:50:19    8628.00
                2020-09-04 23:55:17    8635.80
                ...
    :return: hour_ele: pd.Series
            每小时用电量数据。index为时间，data为用电量：
                index
                2020-09-04 19:00:00    109.80
                2020-09-04 20:00:00    106.20
                2020-09-04 21:00:00    104.40
                2020-09-04 22:00:00    105.00
                2020-09-04 23:00:00     96.00
                ...
    """

    if ep_raw.dropna().shape[0] < 2:  # 有效电能数据点少于2个，无法计算用电量，返回None（情况1）
        return
    else:
        ep = ep_raw.copy()
        ep = fix_ep(ep)  # 修正电能数据

        if ep is None:  # 如果存在负向电能过量的情况，无法计算用电量，返回None（情况2）
            return
        else:
            ep = ep.resample("5T").mean().interpolate("linear", limit=12)  # 对1小时内的缺失值做插值
            ele = -ep.diff(periods=-1).dropna().resample("1H").sum()  # 计算每小时用电量

            ep_size = ep.dropna().resample("1H").size()  # 计算每小时有效数据点个数
            notna_idx = ep_size.loc[ep_size == 12].index  # 取有效数据点为12个（即每5分钟都有数据）的小时数
            hour_ele = ele.loc[notna_idx]
            return hour_ele

