#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django.contrib.auth.models import AbstractUser
from django.db import models
from django.urls import reverse
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from common.utils.choices import ROLE_KIND, USER_STATUS, CERT_TYPE
from common.utils.models import BaseModel
from console.organizations.models import Organization


@python_2_unicode_compatible
class Accounts(AbstractUser):
    update_time = models.DateTimeField(auto_now=True)
    create_time = models.DateTimeField(auto_now_add=True)
    certification_status = models.BooleanField(default=False, help_text="认证状态")
    account_type = models.CharField(max_length=32, help_text="用户类型")
    head_img = models.CharField(max_length=32, help_text="头像", null=True)

    def __str__(self):
        return self.username

    def get_absolute_url(self):
        return reverse('users:detail', kwargs={'username': self.username})

    class Meta:
        db_table = _('accounts')


class Users(BaseModel):
    account = models.OneToOneField(to=Accounts,
                                   on_delete=models.SET_NULL,
                                   null=True,
                                   related_name="user_account")
    nickname = models.CharField(max_length=64, null=True, blank=True)
    wechat = models.CharField(max_length=64, null=True, blank=True)
    phone = models.CharField(max_length=16, null=True, blank=True)
    email = models.CharField(max_length=64, null=True, blank=True)
    code = models.CharField(max_length=64, null=True, blank=True, help_text="工号")
    # 0 女 1 男
    gender = models.BooleanField(default=1, null=True, blank=True, help_text="性别")
    cert = models.IntegerField(choices=CERT_TYPE, default=1, help_text="证件类型")
    cert_number = models.CharField(max_length=64, null=True, blank=True, help_text="证件编号")
    status = models.IntegerField(choices=USER_STATUS, default=1, help_text="人员状态")
    order_rec_setting = models.NullBooleanField(default=None, help_text="接单设置")
    order_rec_status = models.NullBooleanField(default=None, help_text="接单状态")
    is_manager = models.BooleanField(default=False, help_text="是否是公司管理层")

    class Meta:
        db_table = _("users")


class Enterprise(BaseModel):
    enterprise_id = models.CharField(max_length=32,
                                     null=True,
                                     blank=True,
                                     unique=True,
                                     help_text="公司id")
    name = models.CharField(max_length=64,
                            null=True,
                            blank=True,
                            help_text="公司名称")
    logo = models.CharField(max_length=255,
                            null=True,
                            blank=True,
                            help_text="企业logo")
    file = models.CharField(max_length=255,
                            null=True,
                            blank=True,
                            help_text="附件")
    address = models.CharField(max_length=1024,
                               null=True,
                               blank=True,
                               help_text="公司地址")
    adcode = models.CharField(max_length=16,
                              null=True,
                              blank=True,
                              help_text="省级编码")
    city_code = models.CharField(max_length=16,
                                 null=True,
                                 blank=True,
                                 help_text="城市编码")
    district = models.CharField(max_length=16,
                                null=True,
                                blank=True,
                                help_text="区县编码")
    longitude = models.CharField(max_length=32,
                                 null=True,
                                 blank=True,
                                 help_text="经度")
    latitude = models.CharField(max_length=32,
                                null=True,
                                blank=True,
                                help_text="纬度")

    unit_type = models.SmallIntegerField(default=0, help_text="企业类型（对内）", null=True)
    level = models.SmallIntegerField(default=0, help_text="客户等级", null=True)

    legal_person = models.CharField(max_length=32,
                                    null=True,
                                    blank=True,
                                    help_text="法人")
    legal_person_phone = models.CharField(max_length=16,
                                          null=True,
                                          blank=True,
                                          help_text="法人联系方式")

    tax_num = models.CharField(max_length=64,
                               null=True,
                               blank=True,
                               help_text="税号")
    bank_account_name = models.CharField(max_length=255,
                                         null=True,
                                         blank=True,
                                         help_text="银行开户名")
    bank_name = models.CharField(max_length=255,
                                 null=True,
                                 blank=True,
                                 help_text="银行名")

    postcode = models.CharField(max_length=16,
                                null=True,
                                blank=True,
                                help_text="邮编")
    scale = models.IntegerField(null=True, blank=True, help_text="企业规模")
    nature = models.SmallIntegerField(null=True, blank=True, help_text="企业性质")
    industry = models.IntegerField(null=True, blank=True, help_text="所属行业")
    asset = models.IntegerField(null=True, blank=True, help_text="资产规模")
    covered_area = models.IntegerField(null=True, blank=True, help_text="占地面积")

    admin = models.ForeignKey(to="Users",
                              on_delete=models.SET_NULL,
                              null=True,
                              help_text="管理员")

    is_subsidiary = models.BooleanField(default=False, help_text="是否为附属公司")
    parent_enterprise = models.ForeignKey(to="self",
                                          on_delete=models.SET_NULL,
                                          null=True,
                                          help_text="上级公司id")
    is_admin = models.BooleanField(default=False, help_text="是否为管理员公司")

    class Meta:
        db_table = _("enterprise")


class UserEnterprise(BaseModel):
    user = models.ForeignKey(to=Users,
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="ent_user")
    enterprise = models.ForeignKey(
        to=Enterprise,
        on_delete=models.SET_NULL,
        null=True,
        to_field="enterprise_id",
        related_name="user_ent",
    )
    join_date = models.DateField(null=True, blank=True, help_text="入职时间")

    class Meta:
        db_table = _("user_enterprise")


class Roles(BaseModel):
    name = models.CharField(max_length=64, unique=True, help_text="角色名称")
    kind = models.CharField(max_length=32, choices=ROLE_KIND, help_text="所属组织")
    kind_id = models.CharField(max_length=64, help_text=u'角色所属范围id')
    comments = models.CharField(max_length=256, default="", help_text=u'角色描述')

    class Meta:
        db_table = _("roles")


class UserRoles(BaseModel):
    user = models.ForeignKey(to=Users,
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="role_user")
    role = models.ForeignKey(to=Roles,
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="user_role")

    class Meta:
        db_table = _("user_roles")


class Perms(BaseModel):
    name = models.CharField(max_length=64, unique=True, help_text="权限名称")
    code = models.IntegerField(unique=True, help_text="权限编码")
    kind = models.CharField(max_length=32, help_text=u'权限所属')
    group = models.CharField(max_length=32, help_text=u'权限分组')
    desc = models.CharField(max_length=32, help_text=u'权限描述')

    class Meta:
        db_table = _("perms")


class RolePerms(BaseModel):
    perms = models.ForeignKey(to=Perms,
                              on_delete=models.SET_NULL,
                              null=True,
                              related_name="role_perms")
    role = models.ForeignKey(to=Roles,
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="perms_role")

    class Meta:
        db_table = _("role_perms")


class Tags(BaseModel):
    name = models.CharField(max_length=64, help_text=u'标签名称')

    class Meta:
        db_table = _("tags")


class UserTags(BaseModel):
    user = models.ForeignKey(to=Users,
                             help_text=u'用户id',
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="tags_user")
    tag = models.ForeignKey(to=Tags,
                            help_text=u'标签id',
                            on_delete=models.SET_NULL,
                            null=True,
                            related_name="user_tags")

    class Meta:
        db_table = _("user_tags")


class UserOrganizations(BaseModel):
    user = models.ForeignKey(to=Users,
                             help_text=u'用户id',
                             on_delete=models.SET_NULL,
                             null=True,
                             related_name="organization_user")
    organization = models.ForeignKey(to=Organization,
                                     help_text=u'组织id',
                                     on_delete=models.SET_NULL,
                                     null=True,
                                     related_name="user_organization")

    class Meta:
        db_table = _("user_organizations")


class OrganizationRoles(BaseModel):
    organization = models.ForeignKey(to=Organization,
                                     help_text=u'组织id',
                                     on_delete=models.SET_NULL,
                                     null=True)
    role = models.ForeignKey(to=Roles,
                             on_delete=models.SET_NULL,
                             null=True)

    class Meta:
        db_table = _("organization_roles")
