import React from 'react';
import { connect } from 'dva';
import ReactEchartsCore from 'echarts-for-react/lib/core';
import echarts from 'echarts/lib/echarts';
import { useRequest } from 'umi';
import get from 'lodash/get';
import moment from 'moment';
import isEmpty from 'lodash/isEmpty';
import * as services from '../../services';
import Container from '../../components/Container';

const areaColor = [
  'rgba(255, 64, 65, .2)',
  'rgba(14, 244, 245, .2)',
  'rgba(255, 174, 50, .2)',
  'rgba(12, 235, 163, .2)',
];

const getOption = (data = [], config = {}) => {
  const date = get(data, '[0].time', []);
  const startDate = moment(date[0]);
  const endDate = moment(date[date.length - 1]);
  const diff = endDate.diff(startDate, 'days');
  const option = {
    legend: {
      bottom: 20,
      right: 10,
      itemGap: 40,
      itemWidth: 10,
      itemHeight: 10,
      icon: 'circle',
      textStyle: {
        color: '#ffffff',
      },
    },
    color: [
      'rgba(255, 64, 65, .8)',
      'rgba(14, 244, 245, .8)',
      'rgba(255, 174, 50, .8)',
      'rgba(12, 235, 163, .8)',
    ],
    tooltip: {
      trigger: 'axis',
      // formatter: '{b0}: {c0}<br />{b1}: {c1}'
      // formatter: tooltipFormatter,
      // axisPointer: {
      //   lineStyle: {
      //     color: '#222E34',
      //   },
      // },
    },
    grid: {
      left: '2%',
      right: 40,
      bottom: 50,
      containLabel: true,
    },
    xAxis: {
      type: 'time',
      name: '时间',
      boundaryGap: false,
      minInterval: 8,
      splitLine: {
        show: false,
      },
      axisLine: {
        lineStyle: {
          color: '#ffffff',
        },
      },
      axisLabel: {
        showMaxLabel: false,
        // formatter: '${yyyy}-${MM}-${dd}'
        // interval: function(index, value) {
        //   return diff
        // },
        // rotate: 90,
        // formatter: '{yyyy}-{MM}-{dd}'
        // formatter: function(value, index) {
        //   const momentValue = moment(value)
        //   // 大于 5 天只显示 月-日
        //   // if (diff > 5) {
        //   //   return `${momentValue.format('MM-DD')}`
        //   // }
        //   if (momentValue.hours() === 0) {
        //     return `${momentValue.format('MM-DD')}`
        //   }
        //   return `${moment(value).format('HH:mm')}`
        // }
      },
    },
    yAxis: {
      type: 'value',
      name: 'kWh',
      axisLine: {
        lineStyle: {
          color: '#ffffff',
        },
      },
      splitLine: {
        lineStyle: {
          color: '#222E34',
        },
      },
      ...get(config, 'yAxis', {}),
    },
    series: data.map((item, index) => ({
      type: 'line',
      showSymbol: false,
      smooth: true,
      areaStyle: {
        color: areaColor[index],
      },
      ...item,
    })),
  };
  return option;
};

export default connect(({ screen }) => ({
  stationData: screen.stationData,
}))(
  React.memo(function ChartLine(props) {
    const {
      number,
      stationId,
      point,
      startTime,
      endTime,
      load,
      unit = '',
      fields = [],
    } = props;
    const query = fields.map(item => `&value=${item.value}`).join('');
    const { data, loading } = useRequest(
      () => {
        if (!startTime || !endTime || !load) {
          return '';
        }
        return services.getMonitorData({
          number,
          stationId,
          point,
          startTime,
          endTime,
          query,
        });
      },
      {
        formatResult(res) {
          return get(res, 'data.list', []);
        },
        refreshDeps: [point, startTime, endTime, load],
        ready: point && startTime && endTime,
      },
    );

    const chartData = fields.map(item => {
      return {
        data: data
          ? data.map((i, index) => {
              const time = moment(i.tm).format('YYYY-MM-DD HH:mm:ss');
              return [time, i[item.value]];
            })
          : [],
        ...item,
      };
    });
    console.log(chartData);
    return (
      <Container loading={loading} empty={isEmpty(data)}>
        <ReactEchartsCore
          option={getOption(chartData, { yAxis: { name: unit } })}
          echarts={echarts}
          notMerge={true}
          lazyUpdate={true}
          style={{ width: '766px', height: '365px' }}
        />
      </Container>
    );
  }),
);
