from rest_framework import status
from rest_framework.response import Response
from django.views.decorators.gzip import gzip_page
from django.utils.decorators import method_decorator

from common.utils.views import BaseApiView
from common.utils.views import general_message

from crm.crm_customer.serializers import (CrmCustomerClueSerializer, CrmCustomerClueUSerializer,
                                          CrmCustomerClueApproveSerializer, CustomerRemarkSerializer,
                                          CustomerRemarkInfoSerializer)
from crm.crm_customer.services import crm_customer_service

from crm.crm_customer.filters import CustomerRemarkFilter
from crm.crm_customer.models import CustomerRemark

from common.utils.return_structure import error_message


class CrmCustomerLView(BaseApiView):
    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        page = request.GET.get("page", 1)
        page_size = request.GET.get("page_size", 10)
        rst, count = crm_customer_service.get_customers(self.enterprise_ids, self.user.id, **query)
        data = general_message(list=rst, count=count, page=page, page_size=page_size)
        return Response(data, status=status.HTTP_200_OK)

class CrmCustomerDistributeUView(BaseApiView):
    def put(self, request, customer_id, *args, **kwargs):
        salesman_id = request.data.get('salesman_id')
        crm_customer_service.distribute_customer(customer_id, salesman_id)
        return Response(general_message(msg_show="分配成功"), status=status.HTTP_200_OK)


@method_decorator(gzip_page, name='dispatch')
class AppCrmCustomerLView(BaseApiView):
    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        rst, count = crm_customer_service.get_customers_app(self.enterprise_ids, self.user.id, **query)
        data = general_message(list=rst, count=count)
        return Response(data, status=status.HTTP_200_OK)


class CrmCustomerClueLCView(BaseApiView):
    def post(self, request, *args, **kwargs):
        serializer = CrmCustomerClueSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        clue = crm_customer_service.create_customer_clue(self.enterprise.id, self.user.id, serializer.data)
        return Response(general_message("创建成功", bean={"id": clue.id}), status=status.HTTP_200_OK)

    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        page = request.GET.get("page", 1)
        page_size = request.GET.get("page_size", 10)
        rst, count = crm_customer_service.get_customer_clue(self.enterprise_ids, **query)
        data = general_message(list=rst, count=count, page=page, page_size=page_size)
        return Response(data, status=status.HTTP_200_OK)


class CrmCustomerClueRUDView(BaseApiView):
    def get(self, request, clue_id, *args, **kwargs):
        query = request.GET.copy()
        rst = crm_customer_service.get_customer_clue_info(self.enterprise_ids, self.user.id, clue_id, **query)
        data = general_message(bean=rst)
        return Response(data, status=status.HTTP_200_OK)

    def put(self, request, clue_id, *args, **kwargs):
        serializer = CrmCustomerClueUSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        clue = crm_customer_service.update_customer_clue(self.enterprise_ids, self.user.id, serializer.data)
        return Response(general_message("更新成功", bean=clue), status=status.HTTP_200_OK)

    def delete(self, request, clue_id, *args, **kwargs):
        crm_customer_service.del_customer_clue(self.enterprise_ids, self.user.id, clue_id)
        return Response(general_message("删除成功"), status=status.HTTP_200_OK)


class CrmCustomerClueApproveView(BaseApiView):
    def post(self, request, clue_id, *args, **kwargs):
        serializer = CrmCustomerClueApproveSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        salesman_id = serializer.data.get("salesman_id")
        service_enterprise_id = request.data.get("service_enterprise_id")
        crm_customer_service.approve_customer_clue(self.enterprise_ids, clue_id, service_enterprise_id, salesman_id)
        return Response(general_message("审批成功"), status=status.HTTP_200_OK)


class CrmCustomerPortraitView(BaseApiView):
    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        rst = crm_customer_service.get_customer_portrait(self.enterprise.id, **query)
        return Response(general_message(list=rst), status=status.HTTP_200_OK)


class CustomerRemarkView(BaseApiView):
    queryset = CustomerRemark.objects.filter()
    serializer_class = CustomerRemarkInfoSerializer
    filter_class = CustomerRemarkFilter

    def post(self, request, *args, **kwargs):
        request.data.update({"user_id": self.user.id})
        serializer = CustomerRemarkSerializer(data=request.data, context={"request": request})
        if serializer.is_valid():
            serializer.save()
            return Response(general_message("添加成功"))
        return Response(error_message(code=400, msg=serializer.errors,
                                      msg_show="添加失败"))

    def put(self, request, *args, **kwargs):
        instance = self.get_queryset().filter(id=kwargs.get("id")).first()
        if not instance:
            return Response(error_message(code=400, msg_show="备注不存在"))
        serializer = CustomerRemarkSerializer(instance, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(general_message("修改成功"))
        return Response(error_message(code=400, msg_show="修改失败", msg=serializer.errors))

    def get(self, request, *args, **kwargs):
        id = kwargs.get("id")
        if id:
            queryset = self.get_queryset().filter(id=id).first()
            serializer = self.get_serializer(queryset)
            return Response(general_message(bean=serializer.data))
        queryset = self.filter_queryset(self.get_queryset())
        serializer = self.get_serializer(queryset, many=True)
        return Response(general_message(list=serializer.data))

    def delete(self, request, *args, **kwargs):
        CustomerRemark.objects.filter(id=kwargs.get("id")).delete()
        return Response(general_message("删除成功"))


class CrmCustomerRemarkLView(BaseApiView):
    def get(self, request, customer_id, *args, **kwargs):
        rst = crm_customer_service.get_customer_remarks(customer_id)
        dt = general_message(list=rst)
        return Response(dt, status=status.HTTP_200_OK)
