import json
import logging

from django.db.transaction import atomic
from common.utils.models import BaseConnection
from common.utils.exceptions import ValidationError
from crm.crm_customer.repositories import crm_customer_repo
from crm.crm_customer.models import CustomerClue

logger = logging.getLogger("")
base_db = BaseConnection()


class CrmCustomerService(object):
    def get_customers(self, enterprise_id, user_id, **kwargs):
        return crm_customer_repo.get_customers(enterprise_id, user_id, **kwargs)

    def get_customers_app(self, enterprise_id, user_id, **kwargs):
        return crm_customer_repo.get_customers_app(enterprise_id, user_id, **kwargs)

    def get_customer_clue(self, enterprise_ids, **kwargs):
        return crm_customer_repo.get_customer_clue(enterprise_ids, **kwargs)

    def get_customer_clue_info(self, enterprise_id, user_id, clue_id, **kwargs):
        return crm_customer_repo.get_customer_clue_info(enterprise_id, user_id, clue_id, **kwargs)

    def get_customer_portrait(self, enterprise_id, **kwargs):
        return crm_customer_repo.get_customer_portrait(enterprise_id, **kwargs)

    def create_customer_clue(self, enterprise_id, user_id, dt):
        dt["salesman_id"] = user_id
        dt["inputer_id"] = user_id
        dt["enterprise_id"] = enterprise_id
        dt["content"] = json.dumps(dt["content"])
        return crm_customer_repo.create_customer_clue(dt)

    def update_customer_clue(self, enterprise_ids, user_id, dt):
        clue_id = dt["id"]
        dt["salesman_id"] = user_id
        dt["content"] = json.dumps(dt["content"])
        crm_customer_repo.update_customer_clue(dt)
        return self.get_customer_clue_info(enterprise_ids, user_id, clue_id)

    def del_customer_clue(self, enterprise_ids, user_id, clue_id):
        return crm_customer_repo.del_customer_clue(enterprise_ids, user_id, clue_id)

    @atomic()
    def approve_customer_clue(self, enterprise_ids, clue_id, service_enterprise_id, salesman_id=None):
        clue = CustomerClue.objects.filter(enterprise_id__in=enterprise_ids, id=clue_id).first()
        if not clue:
            raise ValidationError(msg_show="客户线索不存在")
        if clue.status == 1:
            raise ValidationError(msg_show="客户线索已审批")
        content = json.loads(clue.content)
        customer_payload = content.pop("base", None)
        crm_customer_payload = {
            "level": clue.level,
            "salesman_id": salesman_id,
        }
        ent_payload = content.pop("enterprise", None)
        ent_payload.pop("streetscape_img", None)
        contact_payload = content.pop("contact", None)
        ele_user_payload = content.pop("ele_user", None)
        enterprise = crm_customer_repo.create_enterprise(**ent_payload)
        customer_payload["enterprise_id"] = enterprise.id
        customer_payload["service_enterprise_id"] = service_enterprise_id
        customer_payload["last_service_staff_id"] = salesman_id
        customer_payload["service_staff_id"] = clue.salesman_id
        customer = crm_customer_repo.create_customer(**customer_payload)
        crm_customer_payload["customer_id"] = customer.id
        crm_customer = crm_customer_repo.create_crm_customer(**crm_customer_payload)
        crm_customer_repo.create_contact(contact_payload)
        crm_customer_repo.create_ele_user(ele_user_payload)
        clue.crm_customer_id = crm_customer.id
        clue.customer_id = customer.id
        clue.status = 1
        clue.save()

    @atomic()
    def distribute_customer(self, customer_id, salesman_id):
        crm_customer_repo.distribute_customer(customer_id, salesman_id)

    def get_customer_remarks(self, customer_id):
        return crm_customer_repo.get_customer_remarks(customer_id)


crm_customer_service = CrmCustomerService()
