from django.db import models

from common.utils.models import BaseModel
from console.customer.models import Customer
from console.electricityuser.models import ElectricityUser

from common.accounts.models import Enterprise


class PowerStation(BaseModel):
    """PowerStation Model"""

    id = models.AutoField(primary_key=True)
    name = models.CharField(max_length=64,
                            null=False,
                            blank=False,
                            help_text='电站名称')
    addr = models.CharField(max_length=256,
                            null=False,
                            blank=False,
                            help_text='电站地址')
    city_code = models.CharField(max_length=32,
                                 null=True,
                                 blank=True,
                                 help_text='城市编码')
    ad_code = models.CharField(max_length=32,
                               null=True,
                               blank=True,
                               help_text='区域编码')
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text='所属客户')
    electricity_user = models.ForeignKey(to=ElectricityUser,
                                         on_delete=models.SET_NULL,
                                         null=True,
                                         help_text='所属户号')
    status = models.BooleanField(default=True, help_text='运行状态')
    operation_level = models.CharField(max_length=16,
                                       null=True,
                                       blank=True,
                                       help_text='运行等级')
    inspections_number = models.IntegerField(null=True, help_text='巡检次数')

    person = models.CharField(max_length=64,
                              null=True,
                              blank=True,
                              help_text='联系人')
    phone = models.CharField(max_length=16,
                             null=True,
                             blank=True,
                             help_text='联系人联络方式')
    file = models.CharField(max_length=256,
                            null=True,
                            blank=True,
                            help_text='一次线路图')
    file_id = models.CharField(max_length=32, null=True, help_text="一次线路图id")
    start_time = models.DateTimeField(null=True, help_text="开始托管时间")
    end_time = models.DateTimeField(null=True, help_text="结束托管时间")
    total_capacity = models.FloatField(default=0, help_text='总容量')
    real_capacity = models.FloatField(default=0, help_text='实际容量')
    customize_time = models.DateField(null=True, help_text="定制化日期")
    electric_test_time = models.DateField(null=True, help_text="电试日期")
    inspection_template = models.IntegerField(null=True, help_text="巡检模板id")
    # 巡检时间格式为list的json字符串，list格式为["00:00","12:00","18:00"],元素为24小时制的时分字符串
    inspection_time = models.CharField(max_length=256,
                                       null=True,
                                       help_text="巡检时间")
    INSPECT_TYPE = ((0, "月巡检"), (1, "日巡检"))
    inspection_type = models.SmallIntegerField(choices=INSPECT_TYPE,
                                               default=0,
                                               help_text="巡检模式")
    service_enterprise = models.ForeignKey(
        to=Enterprise,
        on_delete=models.SET_NULL,
        related_name="station_service_enterprise",
        null=True,
        help_text="所属管理公司")
    inspector_ids = models.CharField(null=True,
                                     max_length=256,
                                     help_text="服务班组id")

    service_team = models.CharField(max_length=64,
                                    null=True,
                                    blank=True,
                                    help_text='服务班组')
    location = models.CharField(max_length=64, null=True, help_text="坐标")
    is_stop = models.BooleanField(default=False, help_text="是否停止")

    class Meta:
        db_table = 'power_station'

    def __str__(self):
        return self.name


class Transformer(BaseModel):
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="所属电站")
    number = models.IntegerField(default=1, help_text="变压器编号")
    capacity = models.FloatField(default=1, help_text="变压器容量")
    real_capacity = models.FloatField(default=0, help_text="实际容量")
    comments = models.CharField(max_length=256, null=True, help_text="备注")
    ele_number = models.CharField(max_length=32, null=True, help_text="户号")

    class Meta:
        db_table = "transformer"


class Outline(BaseModel):
    name = models.CharField(max_length=128,
                            null=True,
                            blank=True,
                            help_text='出线侧名称')

    powerstation = models.ForeignKey(to=PowerStation,
                                     on_delete=models.SET_NULL,
                                     null=True,
                                     help_text='所属电站')
    power_number = models.CharField(max_length=16,
                                    null=True,
                                    help_text="所属电源编号")

    class Meta:
        db_table = 'outline'

    def __str__(self):
        return self.name


class Circuit(BaseModel):
    draw = models.TextField(null=True, blank=True, help_text="压缩后的一次线路图")
    power_station = models.ForeignKey(to=PowerStation,
                                      on_delete=models.SET_NULL,
                                      null=True,
                                      help_text='所属电站')

    class Meta:
        db_table = "power_station_circuit"
