from django.db import models

# Create your models here.
from common.accounts.models import Users
from common.utils.models import BaseModel
from console.task.models import Task
from console.team.models import Team

from console.customer.models import Customer

from console.powerstation.models import PowerStation


class Order(BaseModel):
    name = models.CharField(max_length=64, help_text="名称")
    task = models.ForeignKey(to=Task,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="所属任务")
    recipient = models.ForeignKey(to=Users,
                                  on_delete=models.SET_NULL,
                                  null=True,
                                  help_text="接单人")
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="班组")
    content = models.TextField(help_text="内容", null=True)
    commencement_date = models.DateTimeField(null=True, help_text="施工日期")
    file = models.TextField(null=True, help_text="施工图片")
    conclusion = models.TextField(null=True, help_text="结论")
    receiving_time = models.DateTimeField(null=True, help_text="接单时间")
    status = models.CharField(max_length=32, help_text="状态")
    type = models.CharField(max_length=32, help_text="类型")
    start_time = models.DateTimeField(null=True, help_text="开始时间")
    finish_time = models.DateTimeField(null=True, help_text="结束时间")
    handler = models.ForeignKey(to=Users,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="处理人",
                                related_name="handler")
    transfer_reason = models.CharField(max_length=256,
                                       null=True,
                                       help_text="转单理由")

    class Meta:
        db_table = "order"


class WorkTicket(BaseModel):
    order = models.ForeignKey(to=Order,
                              on_delete=models.SET_NULL,
                              null=True,
                              help_text="工单")
    code = models.CharField(max_length=32, help_text="工作票编号")
    type = models.SmallIntegerField(help_text="工作票类型", default=1)
    customer = models.ForeignKey(to=Customer,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="客户")
    station = models.ForeignKey(to=PowerStation,
                                on_delete=models.SET_NULL,
                                null=True,
                                help_text="电站")
    addr = models.CharField(max_length=256, null=True, help_text="地址")
    job_content = models.TextField(null=True, help_text="工作内容")
    plan_start_time = models.DateTimeField(null=True, help_text="计划开工时间")
    plan_end_time = models.DateTimeField(null=True, help_text="计划完工时间")
    safety_measure = models.TextField(null=True, help_text="安全措施")
    file = models.CharField(max_length=256, null=True, help_text="一次线路图")
    issuer = models.ForeignKey(to=Users,
                               on_delete=models.SET_NULL,
                               null=True,
                               help_text="签发人",
                               related_name="issuer")
    issue_time = models.DateTimeField(null=True, help_text="签发时间")
    person_liable = models.ForeignKey(to=Users,
                                      on_delete=models.SET_NULL,
                                      null=True,
                                      related_name="person_liable",
                                      help_text="负责人")
    team = models.ForeignKey(to=Team,
                             on_delete=models.SET_NULL,
                             null=True,
                             help_text="执行班组")
    weather = models.CharField(max_length=32, null=True, help_text="天气")
    supplement = models.TextField(null=True, help_text="补充内容")
    person_num = models.IntegerField(null=True, help_text="人数")
    person_name = models.CharField(max_length=256, null=True, help_text="姓名")
    delay = models.DateTimeField(null=True, help_text="延期")
    delay_leader = models.CharField(max_length=16,
                                    null=True,
                                    help_text="延期负责人")
    delay_licensor = models.CharField(max_length=16,
                                      null=True,
                                      help_text="延期许可人")
    finished_time = models.DateTimeField(null=True, help_text="结束时间")
    checked = models.TextField(null=True, help_text="工作票检查")
    remarks = models.TextField(null=True, help_text="备注")
    the_end = models.TextField(null=True, help_text="工作票终结")
    work_conditions = models.TextField(null=True, help_text="工作条件")
    attention = models.TextField(null=True, help_text="注意事项")
    supplement_safety_measure = models.CharField(max_length=1024,
                                                 null=True,
                                                 help_text="补充安全措施")
    last_leader = models.CharField(max_length=16, null=True, help_text="原负责人")
    leader_change_issuer = models.CharField(max_length=16,
                                            null=True,
                                            help_text="负责人变更工作许可人")
    leade_change_date = models.DateTimeField(null=True, help_text="负责人变更时间")
    allow_start_time = models.DateTimeField(null=True, help_text="允许开工时间")
    allower = models.CharField(max_length=16, null=True, help_text="允许开工负责人")
    delay_date = models.DateTimeField(null=True, help_text="延期日期")
    other = models.CharField(null=True, max_length=1024, help_text="其他事项")
    handover_confirm = models.CharField(max_length=256,
                                        null=True,
                                        help_text="交接安全确认")
    nameplate_operator = models.CharField(max_length=16,
                                          null=True,
                                          help_text="铭牌操作员")
    nameplate_supervisor = models.CharField(max_length=16,
                                            null=True,
                                            help_text="铭牌监督员")
    supervisor = models.CharField(max_length=16, null=True, help_text="监护人")
    supervisor_event = models.CharField(max_length=1024,
                                        null=True,
                                        help_text="监护事件")

    class Meta:
        db_table = "work_ticket"


class OrderReport(BaseModel):
    order = models.ForeignKey(to=Order,
                              on_delete=models.SET_NULL,
                              null=True,
                              help_text="所属订单")
    problem = models.CharField(max_length=256,
                               help_text="事故经过及原因分析",
                               null=True)
    describe = models.TextField(help_text="事故描述", null=True)
    investigation = models.TextField(help_text="事故排查", null=True)
    handle = models.TextField(help_text="事故处理", null=True)
    proposal = models.TextField(help_text="反制措施及建议", null=True)
    operator = models.ForeignKey(to=Users,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="操作人")
    is_finish = models.BooleanField(default=False, help_text="任务是否完成标记")

    class Meta:
        db_table = "order_record"


class Material(BaseModel):
    name = models.CharField(max_length=32, null=True, help_text="名称")
    code = models.CharField(max_length=32, null=True, help_text="编码")
    specification = models.CharField(max_length=32, null=True, help_text="型号")
    price = models.DecimalField(max_digits=12,
                                decimal_places=2,
                                default=0,
                                help_text="单价")

    class Meta:
        db_table = "material"


class ConsumableMaterial(BaseModel):
    order = models.ForeignKey(to=Order,
                              on_delete=models.SET_NULL,
                              null=True,
                              help_text="所属工单")
    material = models.ForeignKey(to=Material,
                                 on_delete=models.SET_NULL,
                                 null=True,
                                 help_text="物料")
    number = models.IntegerField(default=0, help_text="数量")
    self_provided = models.BooleanField(default=False, help_text="是否自备")

    class Meta:
        db_table = "consumable_material"


class PowerLog(BaseModel):
    order = models.ForeignKey(to=Order,
                              on_delete=models.SET_NULL,
                              null=True,
                              help_text="所属工单")
    time = models.DateTimeField(null=True, help_text="时间")
    content = models.TextField(null=True, help_text="内容")
    picture = models.TextField(null=True, help_text="图片")
    file = models.TextField(null=True, help_text="文件")
    file_title = models.CharField(null=True, max_length=32, help_text="文件标题")

    class Meta:
        db_table = "power_log"
