# -*- coding=utf-8 -*-
from rest_framework import status
from rest_framework.response import Response

from common.utils.views import BaseApiView, AllowAnyApiView
from common.utils.exceptions import ValidationError
from common.utils.return_structure import general_message

from console.equipment.services import equipment_service
from console.equipment.repositories import equipment_queryset
from console.equipment.serialisers import EquipmentCSerializer

from console.equipment.models import Equipment
from console.equipment.serialisers import EquimentInfoSerializer


class EquipmentTreeLCView(BaseApiView):
    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        dt = equipment_queryset.equipment_tree(**query)
        data = general_message(list=dt)
        return Response(data, status=status.HTTP_200_OK)

    def post(self, request, *args, **kwargs):
        serializer = EquipmentCSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        electricity_user_id = serializer.data.get("electricity_user_id")
        customer_id = serializer.data.get("customer_id")
        equipment_service.create_equipments(serializer.data)
        dt = equipment_queryset.equipment_tree(**{"electricity_user_id": electricity_user_id,
                                                  "customer_id": customer_id})
        data = general_message(list=dt)
        return Response(data, status=status.HTTP_200_OK)


class EquipmentLCView(BaseApiView):
    def get(self, request, *args, **kwargs):
        query = request.GET.copy()
        if self.is_manager:
            dt, count = equipment_queryset.get_equipments(self.enterprise_ids, **query)
        else:
            dt, count = equipment_queryset.get_equipments([str(self.service_ent_id)], **query)
        data = general_message(list=dt, count=count)
        return Response(data, status=status.HTTP_200_OK)


class EquipmentDataLView(BaseApiView):
    def get(self, request, id, *args, **kwargs):
        bean = {}
        dt = equipment_queryset.get_equipment_data(id)
        if dt:
            dt.id = id
            bean = dt.to_dict()
        data = general_message(bean=bean)
        return Response(data, status=status.HTTP_200_OK)

    def put(self, request, id, *args, **kwargs):
        bean = {}
        serializer = EquipmentCSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        dt = equipment_queryset.create_or_update_equipment_data(id, serializer.data)
        if dt:
            dt.id = id
            bean = dt.to_dict()
        data = general_message(bean=bean)
        return Response(data, status=status.HTTP_200_OK)

    def delete(self, request, id, *args, **kwargs):
        equipment_queryset.delete_equipment_data(id)
        return Response(general_message(), status=status.HTTP_200_OK)


class TransformerLView(BaseApiView):
    def get(self, request, *args, **kwargs):
        electricity_user_id = request.GET.get("electricity_user_id")
        if not electricity_user_id:
            raise ValidationError("缺少参数")
        dt = equipment_queryset.get_eq(electricity_user_id)
        data = general_message(list=dt)
        return Response(data, status=status.HTTP_200_OK)


class EquimentInfoView(AllowAnyApiView):
    queryset = Equipment.objects.all()
    serializer_class = EquimentInfoSerializer

    def get(self, request, *args, **kwargs):
        instance = self.get_queryset().filter(id=kwargs.get("id")).first()
        serializer = self.get_serializer(instance)
        return Response(general_message(bean=serializer.data))
