import json

from rest_framework import serializers as serializer

from console.dictionary.models import Dictionary
from console.equipment.models import Equipment
from console.equipment.serialisers import EquipmentSerializer


class DictionaryCreateSerializer(serializer.Serializer):
    model = serializer.CharField(required=True,
                                 allow_blank=False,
                                 allow_null=False,
                                 help_text="模块")
    name = serializer.CharField(required=True,
                                allow_blank=False,
                                allow_null=False,
                                help_text="名称")
    value = serializer.ListField(required=True,
                                 allow_null=False,
                                 allow_empty=False,
                                 help_text="枚举值")
    equipment = serializer.IntegerField(required=False,
                                        help_text="关联设备",
                                        allow_null=True)
    remarks = serializer.CharField(required=False,
                                   allow_null=True,
                                   allow_blank=True,
                                   help_text="备注")

    def validate(self, attrs):
        if attrs.get("equipment"):
            euipment = Equipment.objects.filter(
                id=attrs.get("euipment")).first()
            if not euipment:
                raise serializer.ValidationError("设备{}不存在".format(
                    attrs.get("euipment")))
        return attrs

    def create(self, validated_data):
        dictionary = Dictionary.objects.create(
            model=validated_data.get("model"),
            name=validated_data.get("name"),
            value=json.dumps(validated_data.get("value"), ensure_ascii=False),
            equipment_id=validated_data.get("equipment"),
            remarks=validated_data.get("remarks"))
        return dictionary


class DictionaryUpdateSerializer(serializer.Serializer):
    model = serializer.CharField(required=True,
                                 allow_blank=False,
                                 allow_null=False,
                                 help_text="模块")
    name = serializer.CharField(required=True,
                                allow_blank=False,
                                allow_null=False,
                                help_text="名称")
    value = serializer.ListField(required=True,
                                 allow_null=False,
                                 allow_empty=False,
                                 help_text="枚举值")
    equipment = serializer.IntegerField(required=False,
                                        help_text="关联设备",
                                        allow_null=True)
    remarks = serializer.CharField(required=False,
                                   allow_null=True,
                                   allow_blank=True,
                                   help_text="备注")

    def validate(self, attrs):
        if attrs.get("equipment"):
            equipment = Equipment.objects.filter(
                id=attrs.get("equipment")).first()
            if not equipment:
                raise serializer.ValidationError("设备{}不存在".format(
                    attrs.get("equipment")))
            attrs["equipment"] = equipment
        return attrs

    def update(self, instance, validated_data):
        instance.model = validated_data.get("model")
        instance.name = validated_data.get("name")
        instance.value = validated_data.get("value")
        instance.equipment = validated_data.get("equipment")
        instance.remarks = validated_data.get("remarks")
        instance.save()
        return instance


class DictionaryInfoSerializer(serializer.ModelSerializer):
    value = serializer.SerializerMethodField()
    equipment = serializer.SerializerMethodField()

    def get_value(self, dictionary):
        return json.loads(dictionary.value.replace("'", '"'))

    def get_equipment(self, obj):
        return EquipmentSerializer(obj.equipment).data

    class Meta:
        model = Dictionary
        fields = ('id', 'model', 'name', 'value', 'equipment', 'remarks')
