from functools import reduce
from django.db.models import Q
from django_filters import rest_framework as filter

from common.accounts.models import Enterprise
from common.third_party_api.amap.openapi import AmapApi
from common.utils.exceptions import ValidationError

CUSTOMER_TYPE_CHOICES = ((0, '普通客户'), (1, '签约客户'), (10, '监控客户'))


class CustomerFilter(filter.FilterSet):
    name = filter.CharFilter(method='get_name')
    province = filter.CharFilter(method='get_province')
    city = filter.CharFilter(method='get_city')
    area = filter.CharFilter(method='get_area')
    type = filter.MultipleChoiceFilter(choices=CUSTOMER_TYPE_CHOICES,
                                       method='get_type')

    def get_province(self, queryset, name, value):
        geocodes = AmapApi().geo(address=str(value))['geocodes']
        if not len(geocodes):
            raise ValidationError('地址无效请修改')
        adcode = geocodes[0]['adcode'][0:2]
        if value:
            queryset1 = Enterprise.objects.filter(adcode__startswith=adcode)
            elist = queryset1.values_list("id", flat=True)
            queryset = queryset.filter(enterprise_id__in=elist)
        return queryset

    def get_city(self, queryset, name, value):
        if '城区' in value:
            value = value.replace('城区', '')
            geocodes = AmapApi().geo(address=str(value))['geocodes']
            if not len(geocodes):
                raise ValidationError('地址无效请修改')
            adcode = geocodes[0]['adcode'][0:2]
        else:
            geocodes = AmapApi().geo(address=str(value))['geocodes']
            if not len(geocodes):
                raise ValidationError('地址无效请修改')
            adcode = geocodes[0]['adcode'][0:4]
        if value:
            queryset1 = Enterprise.objects.filter(adcode__startswith=adcode)
            elist = queryset1.values_list("id", flat=True)
            queryset = queryset.filter(enterprise_id__in=elist)
        return queryset

    def get_area(self, queryset, name, value):
        geocodes = AmapApi().geo(address=str(value))['geocodes']
        if not len(geocodes):
            raise ValidationError('地址无效请修改')
        adcode = geocodes[0]['adcode']
        if value:
            queryset1 = Enterprise.objects.filter(adcode=adcode)
            elist = queryset1.values_list("id", flat=True)
            queryset = queryset.filter(enterprise_id__in=elist)
        return queryset

    def get_name(self, queryset, name, value):
        if value:
            queryset = queryset.filter(
                Q(name__contains=value)
                | Q(service_staff__nickname__icontains=value)
                | Q(enterprise__address__icontains=value))
        return queryset

    def get_type(self, queryset, name, value):
        value = reduce(lambda x, y: int(x) + int(y), value)
        return queryset.filter(type=value)


class CustomerReportPDFFilter(filter.FilterSet):
    customer_id = filter.CharFilter(method="get_customer")
    year_month = filter.CharFilter(method="get_yeat_month")

    def get_customer(self, queryset, name, value):
        if value:
            queryset = queryset.filter(customer_id=value)
        return queryset

    def get_yeat_month(self, queryset, name, value):
        if value:
            queryset = queryset.filter(year_month=value)
        return queryset
