-- detect_mime_by_magic.lua
local function detect(filepath)
    local f = io.open(filepath, "rb")
    if not f then return "application/octet-stream" end

    local header = f:read(8)  -- 读前8字节足够判断多数类型
    f:close()

    if not header or #header < 4 then
        return "application/octet-stream"
    end

    -- 常见魔数（Magic Bytes）匹配
    local b1, b2, b3, b4 = string.byte(header, 1, 4)

    -- PNG
    if b1 == 0x89 and b2 == 0x50 and b3 == 0x4E and b4 == 0x47 then
        return "image/png"
    end

    -- JPEG
    if b1 == 0xFF and b2 == 0xD8 and b3 == 0xFF then
        return "image/jpeg"
    end

    -- GIF
    if string.sub(header, 1, 3) == "GIF" then
        return "image/gif"
    end

    -- PDF
    if string.sub(header, 1, 4) == "%PDF" then
        return "application/pdf"
    end

    -- ZIP (包括 JAR, DOCX 等)
    if b1 == 0x50 and b2 == 0x4B and b3 == 0x03 and b4 == 0x04 then
        return "application/zip"
    end

    -- WebP
    if string.sub(header, 1, 4) == "RIFF" and string.sub(header, 7, 8) == "WEBP" then
        return "image/webp"
    end

    -- JSON: 以 { 开头（简单判断）
    if b1 == 0x7B then  -- '{'
        return "application/json"
    end

    -- XML: 以 <?xml 开头
    if string.sub(header, 1, 5) == "<?xml" then
        return "application/xml"
    end

    -- 文本文件（UTF-8 或 ASCII）
    if b1 < 0x80 and b2 < 0x80 and b3 < 0x80 and b4 < 0x80 then
        return "text/plain; charset=utf-8"
    end

    -- 默认未知二进制
    return "application/octet-stream"
end

return detec
